/*
 * dpkg - Debian Package library and the Debian Package Maven plugin
 * (c) Copyright 2016 Gerrit Hohl
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package net.sourceforge.javadpkg.field.impl;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import net.sourceforge.javadpkg.Context;
import net.sourceforge.javadpkg.GlobalConstants;
import net.sourceforge.javadpkg.ParseException;
import net.sourceforge.javadpkg.field.Field;
import net.sourceforge.javadpkg.field.FieldParser;
import net.sourceforge.javadpkg.io.DataSource;


/**
 * <p>
 * A {@link FieldParser} implementation.
 * </p>
 *
 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
 * @version <b>1.0</b>, 03.01.2016 by Gerrit Hohl
 */
public class FieldParserImpl implements FieldParser, GlobalConstants {
	
	
	/** The flag if empty lines should not throw a {@link ParseException}. */
	private boolean	allowEmptyLines;
	/**
	 * <p>
	 * The flag if empty lines should be returned by the
	 * {@link #parseFieldsAsList(DataSource, Context)} method.
	 * </p>
	 */
	private boolean	returnEmptyLines;
	/**
	 * The flag if nameless fields should not throw a {@link ParseException}.
	 */
	private boolean	allowNamelessFields;
	
	
	/**
	 * <p>
	 * Creates a parser.
	 * </p>
	 *
	 * @param allowEmptyLines
	 *            The flag if empty lines are allowed. Otherwise a
	 *            {@link ParseException} is thrown when an empty line is found.
	 * @param returnEmptyLines
	 *            The flag if empty lines should be returned by the
	 *            {@link #parseFieldsAsList(DataSource, Context)} method.
	 * @param allowNamelessFields
	 *            The flag if nameless fields should not throw a
	 *            {@link ParseException}.
	 */
	public FieldParserImpl(boolean allowEmptyLines, boolean returnEmptyLines, boolean allowNamelessFields) {
		super();
		
		this.allowEmptyLines = allowEmptyLines;
		this.returnEmptyLines = returnEmptyLines;
		this.allowNamelessFields = allowNamelessFields;
	}


	@Override
	public Map<String, Field> parseFieldsAsMap(DataSource source, Context context) throws IOException, ParseException {
		Map<String, Field> map;
		List<Field> list;
		String name;


		if (source == null)
			throw new IllegalArgumentException("Argument source is null.");
		if (context == null)
			throw new IllegalArgumentException("Argument context is null.");
		
		// --- Get the fields ---
		list = this.parseFieldsAsList(source, context);
		
		// --- Create a map ---
		map = new LinkedHashMap<>();
		for (Field field : list) {
			// --- Ignore empty fields and nameless fields ---
			if (field.isEmpty() || field.isNameless()) {
				continue;
			}

			name = field.getName().toLowerCase();
			if (map.containsKey(name))
				throw new ParseException("The field |" + field.getName() + "| exists more than one time in the source |"
						+ source.getName() + "|.");
			map.put(name, field);
		}
		return map;
	}


	@Override
	public List<Field> parseFieldsAsList(DataSource source, Context context) throws IOException, ParseException {
		List<Field> fields;
		String line, name, value;
		FieldImpl field = null;
		int index;
		boolean lastLine = false;


		if (source == null)
			throw new IllegalArgumentException("Argument source is null.");
		if (context == null)
			throw new IllegalArgumentException("Argument context is null.");
		
		// --- Parse the fields ---
		fields = new ArrayList<>();
		try {
			try (BufferedReader reader = new BufferedReader(new InputStreamReader(source.getInputStream(), UTF_8_CHARSET))) {
				while ((line = reader.readLine()) != null) {
					// --- Did we already read the last line? ---
					if (lastLine)
						throw new ParseException(
								"Found a line |" + line + "| in source |" + source.getName() + "| which is not a field.");
					
					// --- A comment? ---
					if (line.startsWith("#")) {
						continue;
					}
					// --- A continuation? ---
					else if (line.startsWith(" ") || line.startsWith("\t")) {
						if (field == null)
							if (this.allowNamelessFields) {
								fields.add(new NamelessField(line));
							} else
								throw new ParseException("Found a continuation line |" + line + "| in source |"
										+ source.getName() + "|, but the initial line of the field is missing.");
						else {
							field.addValue("\n");
							line = line.substring(1);
							field.addValue(line);
						}
					}
					// --- Otherwise it must be a new field or an empty line ---
					else {
						// --- Do we have an empty line ---
						if (line.isEmpty()) {
							// --- If we don't allow empty lines than an empty line must be the last line ---
							if (!this.allowEmptyLines) {
								lastLine = true;
							}
							// --- If empty lines should be return we add one ---
							else if (this.returnEmptyLines) {
								fields.add(EmptyField.EMPTY_FIELD);
							}
						}
						// --- Otherwise we should have a new field ---
						else {
							index = line.indexOf(':');
							if (index == -1) {
								if (this.allowNamelessFields) {
									fields.add(new NamelessField(line));
								} else
									throw new ParseException("Found a line |" + line + "| in source |" + source.getName()
											+ "| which is not a field.");
							} else {
								name = line.substring(0, index).trim();
								value = line.substring(index + 1).trim();
								field = new FieldImpl(name, value);
								fields.add(field);
							}
						}
					}
				}
			}
		} catch (IOException e) {
			throw new IOException(
					"Couldn't read control for Debian binary package from source |" + source.getName() + "|: " + e.getMessage(),
					e);
		}
		return fields;
	}


}
