/*
 * dpkg - Debian Package library and the Debian Package Maven plugin
 * (c) Copyright 2016 Gerrit Hohl
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package net.sourceforge.javadpkg.field.impl;

import net.sourceforge.javadpkg.field.Field;

/**
 * <p>
 * A {@link Field} implementation.
 * </p>
 *
 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
 * @version <b>1.0</b>, 28.04.2016 by Gerrit Hohl
 */
public class FieldImpl extends AbstractField {
	
	
	/** The name. */
	private String			name;
	/** The value. */
	private StringBuilder	value;


	/**
	 * <p>
	 * Creates a field.
	 * </p>
	 * <p>
	 * The value won't be formatted.
	 * </p>
	 *
	 * @param name
	 *            The name.
	 * @param value
	 *            The value.
	 * @throws IllegalArgumentException
	 *             If any of the parameters is <code>null</code>.
	 */
	public FieldImpl(String name, String value) {
		this(name, value, false);
	}
	
	
	/**
	 * <p>
	 * Creates a field.
	 * </p>
	 *
	 * @param name
	 *            The name.
	 * @param value
	 *            The value.
	 * @param formatValue
	 *            The flag if the value should be stored as formatted. A
	 *            formatted value contains &quot;.&quot; instead of empty lines.
	 * @throws IllegalArgumentException
	 *             If any of the parameters is <code>null</code>.
	 */
	public FieldImpl(String name, String value, boolean formatValue) {
		super(false, false);
		
		if (name == null)
			throw new IllegalArgumentException("Argument name is null.");
		if (value == null)
			throw new IllegalArgumentException("Argument value is null.");
		
		this.name = name;
		this.value = new StringBuilder();
		if (formatValue) {
			this.value.append(this.formatValue(value));
		} else {
			this.value.append(value);
		}
	}


	/**
	 * <p>
	 * Returns the name.
	 * </p>
	 *
	 * @return The name.
	 */
	@Override
	public String getName() {
		return this.name;
	}
	
	
	/**
	 * <p>
	 * Returns the value.
	 * </p>
	 *
	 * @return The value.
	 */
	@Override
	public String getValue() {
		return this.value.toString();
	}


	/**
	 * <p>
	 * Sets the specified value as the value of the field.
	 * </p>
	 *
	 * @param value
	 *            The value.
	 * @throws IllegalArgumentException
	 *             If the value is <code>null</code>.
	 */
	public void setValue(String value) {
		if (value == null)
			throw new IllegalArgumentException("Argument value is null.");
		
		this.value = new StringBuilder();
		this.value.append(value);
	}


	/**
	 * <p>
	 * Adds the specified value to the value of the field.
	 * </p>
	 *
	 * @param value
	 *            The value which should be added to the field value.
	 */
	public void addValue(String value) {
		if (value == null)
			throw new IllegalArgumentException("Argument value is null.");
		
		this.value.append(value);
	}
	
	
}
