/*
 * dpkg - Debian Package library and the Debian Package Maven plugin
 * (c) Copyright 2016 Gerrit Hohl
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package net.sourceforge.javadpkg.control.impl;

import java.util.Map;
import java.util.TreeMap;

import net.sourceforge.javadpkg.ParseException;
import net.sourceforge.javadpkg.control.PackageVersionRelationOperator;
import net.sourceforge.javadpkg.control.PackageVersionRelationOperatorParser;


/**
 * <p>
 * A {@link PackageVersionRelationOperatorParser} implementation.
 * </p>
 *
 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
 * @version <b>1.0</b>, 01.01.2016 by Gerrit Hohl
 */
public class PackageVersionRelationOperatorParserImpl implements PackageVersionRelationOperatorParser {


	/** The relational operators. */
	private Map<String, PackageVersionRelationOperator> relationOperators;


	/**
	 * <p>
	 * Creates a parser.
	 * </p>
	 */
	public PackageVersionRelationOperatorParserImpl() {
		super();

		this.relationOperators = new TreeMap<>();
		this.addRelationOperator("<<");
		this.addRelationOperator("<=");
		this.addRelationOperator("=");
		this.addRelationOperator(">=");
		this.addRelationOperator(">>");
	}


	/**
	 * <p>
	 * Adds a relation operator to the internal map.
	 * </p>
	 *
	 * @param text
	 *            The text.
	 */
	private void addRelationOperator(String text) {
		this.relationOperators.put(text, new PackageVersionRelationOperatorImpl(text));
	}
	
	
	@Override
	public PackageVersionRelationOperator parsePackageVersionRelationOperator(String value) throws ParseException {
		PackageVersionRelationOperator relationOperator;


		if (value == null)
			throw new IllegalArgumentException("Argument value is null.");
			
		relationOperator = this.relationOperators.get(value);
		if (relationOperator == null)
			throw new ParseException("The relational operator |" + value + "| is not a supported relational operator.");

		return relationOperator;
	}
	
	
	/* **********************************************************************
	 * **********************************************************************
	 * **********************************************************************
	 * **********************************************************************
	 * **********************************************************************
	 */


	/**
	 * <p>
	 * The {@link PackageVersionRelationOperator} implementation of this class.
	 * </p>
	 *
	 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
	 * @version <b>1.0</b>, 01.01.2016 by Gerrit Hohl
	 */
	private class PackageVersionRelationOperatorImpl implements PackageVersionRelationOperator {
		
		
		/** The text. */
		private String text;
		
		
		/**
		 * <p>
		 * Creates a relation operator.
		 * </p>
		 *
		 * @param text
		 *            The text.
		 */
		public PackageVersionRelationOperatorImpl(String text) {
			super();
			
			this.text = text;
		}
		
		
		@Override
		public String getText() {
			return this.text;
		}


	}
	
	
}
