/*
 * dpkg - Debian Package library and the Debian Package Maven plugin
 * (c) Copyright 2016 Gerrit Hohl
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package net.sourceforge.javadpkg.control.impl;

import net.sourceforge.javadpkg.Context;
import net.sourceforge.javadpkg.ParseException;
import net.sourceforge.javadpkg.control.PackageVersion;
import net.sourceforge.javadpkg.control.PackageVersionParser;


/**
 * <p>
 * A {@link PackageVersionParser} implementation.
 * </p>
 *
 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
 * @version <b>1.0</b>, 01.01.2016 by Gerrit Hohl
 */
public class PackageVersionParserImpl implements PackageVersionParser {
	
	
	/** The regular expression for a valid upstream version. */
	private static final String	REGEXP_UPSTREAM_VERSION	= "([0-9]){1}([A-Za-z0-9.+:~])*";
	/** The regular expression for a valid Debian revision. */
	private static final String	REGEXP_DEBIAN_REVISION	= "([A-Za-z0-9.+~-])+";
	// TODO Handle Debian revisions containing a hyphen (-).
	
	
	/**
	 * <p>
	 * Creates a parser.
	 * </p>
	 */
	public PackageVersionParserImpl() {
		super();
	}


	@Override
	public PackageVersion parsePackageVersion(String value, Context context) throws ParseException {
		PackageVersion packageVersion;
		String rest, epoch = null, upstreamVersion, debianRevision = null;
		int index, intValue;
		
		if (value == null)
			throw new IllegalArgumentException("Argument value is null.");
		if (context == null)
			throw new IllegalArgumentException("Argument context is null.");
		
		rest = value;
		
		// --- Epoch ---
		index = rest.indexOf(':');
		if (index != -1) {
			epoch = rest.substring(0, index);
			rest = rest.substring(index + 1);
			if (epoch.startsWith("0") && !epoch.equals("0")) {
				throw new ParseException("Version |" + value + "| contains an invalid epoch |" + epoch
						+ "|: An epoch can't start with 0 except it is 0.");
			}
			try {
				intValue = Integer.parseInt(epoch);
			} catch (NumberFormatException e) {
				throw new ParseException(
						"Version |" + value + "| contains an invalid epoch |" + epoch + "|: The epoch is not a valid integer.");
			}
			if (intValue < 0)
				throw new ParseException(
						"Version |" + value + "| contains an invalid epoch |" + epoch + "|: The epoch is negative.");
		}

		// --- Upstream version ---
		index = rest.indexOf('-');
		if (index == -1) {
			upstreamVersion = rest;
			rest = "";
		} else {
			upstreamVersion = rest.substring(0, index);
			rest = rest.substring(index + 1);
		}
		if (!upstreamVersion.matches(REGEXP_UPSTREAM_VERSION))
			throw new ParseException("Version |" + value + "| contains an invalid upstream version |" + upstreamVersion + "|");
		
		// --- Debian revision ---
		if (!rest.isEmpty()) {
			debianRevision = rest;
			if (!debianRevision.matches(REGEXP_DEBIAN_REVISION))
				throw new ParseException(
						"Version |" + value + "| contains an invalid Debian revision |" + debianRevision + "|");
		}
		
		packageVersion = new PackageVersionImpl(epoch, upstreamVersion, debianRevision);
		return packageVersion;
	}


	/* **********************************************************************
	 * **********************************************************************
	 * **********************************************************************
	 * **********************************************************************
	 * **********************************************************************
	 */


	/**
	 * <p>
	 * The {@link PackageVersion} implementation of this class.
	 * </p>
	 *
	 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
	 * @version <b>1.0</b>, 01.01.2016 by Gerrit Hohl
	 */
	private class PackageVersionImpl implements PackageVersion {
		
		
		/** The epoch (optional). */
		private String	epoch;
		/** The upstream version. */
		private String	upstreamVersion;
		/** The Debian revision (optional). */
		private String	debianRevision;


		/**
		 * <p>
		 * Creates a package version.
		 * </p>
		 *
		 * @param epoch
		 *            The epoch (optional).
		 * @param upstreamVersion
		 *            The upstream version.
		 * @param debianRevision
		 *            The Debian revision (optional).
		 */
		public PackageVersionImpl(String epoch, String upstreamVersion, String debianRevision) {
			super();
			
			this.epoch = epoch;
			this.upstreamVersion = upstreamVersion;
			this.debianRevision = debianRevision;
		}


		@Override
		public String getEpoch() {
			return this.epoch;
		}
		
		
		@Override
		public String getUpstreamVersion() {
			return this.upstreamVersion;
		}
		
		
		@Override
		public String getDebianRevision() {
			return this.debianRevision;
		}
		
		
		@Override
		public String getText() {
			StringBuilder sb;
			
			
			sb = new StringBuilder();
			if (this.epoch != null) {
				sb.append(this.epoch);
				sb.append(':');
			}
			sb.append(this.upstreamVersion);
			if (this.debianRevision != null) {
				sb.append('-');
				sb.append(this.debianRevision);
			}
			return sb.toString();
		}


	}


}
