/*
 * dpkg - Debian Package library and the Debian Package Maven plugin
 * (c) Copyright 2016 Gerrit Hohl
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package net.sourceforge.javadpkg.control.impl;

import java.util.Map;
import java.util.TreeMap;

import net.sourceforge.javadpkg.Context;
import net.sourceforge.javadpkg.ParseException;
import net.sourceforge.javadpkg.control.Architecture;
import net.sourceforge.javadpkg.control.ArchitectureParser;


/**
 * <p>
 * A {@link ArchitectureParser} implementation.
 * </p>
 *
 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
 * @version <b>1.0</b>, 01.01.2016 by Gerrit Hohl
 */
public class ArchitectureParserImpl implements ArchitectureParser {
	
	
	/** The architectures. */
	private Map<String, Architecture> architectures;
	
	
	/**
	 * <p>
	 * Creates a parser.
	 * </p>
	 */
	public ArchitectureParserImpl() {
		super();
		
		this.architectures = new TreeMap<>();
		this.addArchitecture("all");
		this.addArchitecture("source");
		this.addArchitecture("amd64");
		this.addArchitecture("i386");
	}
	
	
	/**
	 * <p>
	 * Adds an architecture.
	 *
	 * @param architecture
	 *            The architecture.
	 */
	private void addArchitecture(String architecture) {
		this.architectures.put(architecture, new ArchitectureImpl(architecture));
	}


	@Override
	public Architecture parseArchitecture(String value, Context context) throws ParseException {
		Architecture architecture;


		if (value == null)
			throw new IllegalArgumentException("Argument value is null.");
		if (context == null)
			throw new IllegalArgumentException("Argument context is null.");
		
		architecture = this.architectures.get(value);
		if (architecture == null) {
			context.addWarning(new ArchitectureUnsupportedWarning(value));
			architecture = new ArchitectureImpl(value);
		}
		
		return architecture;
	}


	/* **********************************************************************
	 * **********************************************************************
	 * **********************************************************************
	 * **********************************************************************
	 * **********************************************************************
	 */


	/**
	 * <p>
	 * The {@link Architecture} implementation of this class.
	 * </p>
	 *
	 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
	 * @version <b>1.0</b>, 01.01.2016 by Gerrit Hohl
	 */
	private class ArchitectureImpl implements Architecture {
		
		
		/** The text. */
		private String text;


		/**
		 * <p>
		 * Creates a architecture.
		 * </p>
		 *
		 * @param text
		 *            The text.
		 */
		public ArchitectureImpl(String text) {
			super();

			this.text = text;
		}
		
		
		@Override
		public String getText() {
			return this.text;
		}
		
		
	}
}
