/*
 * dpkg - Debian Package library and the Debian Package Maven plugin
 * (c) Copyright 2015 Gerrit Hohl
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package net.sourceforge.javadpkg.control;

import java.util.List;

/**
 * <p>
 * The dependency of a package to other packages.
 * </p>
 * <p>
 * See <a href="https://www.debian.org/doc/debian-policy/ch-relationships.html">
 * Chapter 7 - Declaring relationships between packages</a> for further
 * information.
 * </p>
 *
 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
 * @version <b>1.0</b>, 31.12.2015 by Gerrit Hohl
 */
public interface PackageDependency {
	
	
	/*
	 * TODO Split that interface later in a super interface for package dependencies and a sub interface for normal
	 * package dependencies (package, relation operator, version) and conditional package dependencies (condition and list
	 * of normal package dependencies).
	 */
	
	
	/**
	 * <p>
	 * A condition for a package dependency.
	 * </p>
	 *
	 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
	 * @version <b>1.0</b>, 03.01.2016 by Gerrit Hohl
	 */
	public enum Condition {
		/** An or condition. */
		OR
	}


	/**
	 * <p>
	 * Returns the flag if this package dependency is a condition.
	 * </p>
	 * <p>
	 * If the package dependency is a condition the {@link #getCondition()} and
	 * {@link #getConditionPackageDependencies()} methods must be used.
	 * Otherwise the {@link #getPackage()}, {@link #getRelationOperator()} and
	 * {@link #getVersion()} methods must be used.
	 * </p>
	 *
	 * @return The flag: <code>true</code>, if this package dependency is a
	 *         condition, <code>false</code> otherwise.
	 */
	boolean isCondition();


	/**
	 * <p>
	 * Returns the condition.
	 * </p>
	 *
	 * @return The condition or <code>null</code>, if this package dependency is
	 *         a not a condition.
	 * @see #isCondition()
	 */
	Condition getCondition();


	/**
	 * <p>
	 * Returns the package dependencies for the condition.
	 * </p>
	 *
	 * @return The package dependencies or an empty list, if this package
	 *         dependency is a not a condition.
	 * @see #isCondition()
	 */
	List<PackageDependency> getConditionPackageDependencies();
	
	
	/**
	 * <p>
	 * Returns the name of the package.
	 * </p>
	 *
	 * @return The name or <code>null</code>, if this package dependency is a
	 *         condition.
	 * @see #isCondition()
	 */
	PackageName getPackage();


	/**
	 * <p>
	 * Returns the relational operator.
	 * </p>
	 *
	 * @return The relational operator or <code>null</code>, if no version is
	 *         specified or this package dependency is a condition.
	 * @see #isCondition()
	 */
	PackageVersionRelationOperator getRelationOperator();


	/**
	 * <p>
	 * Returns the version.
	 * </p>
	 *
	 * @return The version or <code>null</code>, if no version is specified or
	 *         this package dependency is a condition.
	 * @see #isCondition()
	 */
	PackageVersion getVersion();


}
