/*
 * dpkg - Debian Package library and the Debian Package Maven plugin
 * (c) Copyright 2015 Gerrit Hohl
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package net.sourceforge.javadpkg.control;

import java.util.List;

/**
 * <p>
 * The control of a binary package.
 * </p>
 *
 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
 * @version <b>1.0</b>, 31.12.2015 by Gerrit Hohl
 */
public interface BinaryControl extends Control {
	
	
	/**
	 * <p>
	 * Returns the name of the binary package.
	 * </p>
	 *
	 * @return The name.
	 */
	PackageName getPackage();


	/**
	 * <p>
	 * Returns the section.
	 * </p>
	 * <p>
	 * See <a href=
	 * "https://www.debian.org/doc/debian-policy/ch-controlfields.html#s-f-Section">
	 * 5.6.5 Section</a> for further information.
	 * </p>
	 *
	 * @return The section.
	 */
	Section getSection();


	/**
	 * <p>
	 * Returns the priority.
	 * </p>
	 *
	 * @return The priority.
	 */
	PackagePriority getPriority();


	/**
	 * <p>
	 * The flag if the package is essential for the system and can't be removed.
	 * </p>
	 * <p>
	 * See <a href=
	 * "https://www.debian.org/doc/debian-policy/ch-controlfields.html#s-f-Essential">
	 * 5.6.9 Essential</a> for further information.
	 * </p>
	 *
	 * @return The flag.
	 */
	Boolean getEssential();


	/**
	 * <p>
	 * Returns the packages on which this package depends on.
	 * </p>
	 * <p>
	 * See <a href=
	 * "https://www.debian.org/doc/debian-policy/ch-relationships.html#s-binarydeps">
	 * 7.2 Binary Dependencies - Depends, Recommends, Suggests, Enhances,
	 * Pre-Depends</a> for further information.
	 * </p>
	 *
	 * @return The dependencies or an empty list, if the package doesn't have
	 *         any dependencies.
	 */
	List<PackageDependency> getDepends();
	
	
	/**
	 * <p>
	 * Returns the packages which this package has a strong, but not absolute,
	 * dependency.
	 * </p>
	 * <p>
	 * See <a href=
	 * "https://www.debian.org/doc/debian-policy/ch-relationships.html#s-binarydeps">
	 * 7.2 Binary Dependencies - Depends, Recommends, Suggests, Enhances,
	 * Pre-Depends</a> for further information.
	 * </p>
	 *
	 * @return The dependencies or an empty list, if the package doesn't have
	 *         any dependencies.
	 */
	List<PackageDependency> getRecommends();
	
	
	/**
	 * <p>
	 * Returns the packages which this package suggests to install for enhancing
	 * its functionality.
	 * </p>
	 * <p>
	 * The opposite of the list returned by the {@link #getEnhances()} method.
	 * </p>
	 * <p>
	 * See <a href=
	 * "https://www.debian.org/doc/debian-policy/ch-relationships.html#s-binarydeps">
	 * 7.2 Binary Dependencies - Depends, Recommends, Suggests, Enhances,
	 * Pre-Depends</a> for further information.
	 * </p>
	 *
	 * @return The dependencies or an empty list, if the package doesn't have
	 *         any dependencies.
	 */
	List<PackageDependency> getSuggests();
	
	
	/**
	 * <p>
	 * Returns the packages which are enhanced in their functionality by this
	 * package.
	 * </p>
	 * <p>
	 * The opposite of the list returned by the {@link #getSuggests()} method.
	 * </p>
	 * <p>
	 * See <a href=
	 * "https://www.debian.org/doc/debian-policy/ch-relationships.html#s-binarydeps">
	 * 7.2 Binary Dependencies - Depends, Recommends, Suggests, Enhances,
	 * Pre-Depends</a> for further information.
	 * </p>
	 *
	 * @return The dependencies or an empty list, if the package doesn't have
	 *         any dependencies.
	 */
	List<PackageDependency> getEnhances();
	
	
	/**
	 * <p>
	 * Returns the packages which must be installed before this package is
	 * installed.
	 * </p>
	 * <p>
	 * In comparison to the list returned by the {@link #getDepends()} method
	 * this packages must be completely installed by the time this package is
	 * installed and not allowed to be installed in the same process.
	 * </p>
	 * <p>
	 * See <a href=
	 * "https://www.debian.org/doc/debian-policy/ch-relationships.html#s-binarydeps">
	 * 7.2 Binary Dependencies - Depends, Recommends, Suggests, Enhances,
	 * Pre-Depends</a> for further information.
	 * </p>
	 *
	 * @return The dependencies or an empty list, if the package doesn't have
	 *         any dependencies.
	 */
	List<PackageDependency> getPreDepends();
	
	
	/**
	 * <p>
	 * Returns the packages which will &quot;broken&quot; by this package.
	 * </p>
	 * <p>
	 * This package can't be installed until the other packages are
	 * deconfigured. The broken packages can't be reconfigured afterwards.
	 * <p>
	 * See <a href=
	 * "https://www.debian.org/doc/debian-policy/ch-relationships.html#s-breaks">
	 * 7.3 Packages which break other packages - Breaks</a> for further
	 * information.
	 * </p>
	 *
	 * @return The dependencies or an empty list, if the package doesn't have
	 *         any dependencies.
	 */
	List<PackageDependency> getBreaks();
	
	
	/**
	 * <p>
	 * Returns the packages which will be in conflict with this package.
	 * </p>
	 * <p>
	 * The conflict packages and this package can't be installed on the same
	 * system at the same time.
	 * </p>
	 * <p>
	 * See <a href=
	 * "https://www.debian.org/doc/debian-policy/ch-relationships.html#s-conflicts">
	 * 7.4 Conflicting binary packages - Conflicts</a> for further information.
	 * </p>
	 *
	 * @return The dependencies or an empty list, if the package doesn't have
	 *         any dependencies.
	 */
	List<PackageDependency> getConflicts();
	
	
	/**
	 * <p>
	 * Returns the virtual packages which are provided by this package.
	 * </p>
	 * <p>
	 * See <a href=
	 * "https://www.debian.org/doc/debian-policy/ch-relationships.html#s-virtual">
	 * 7.5 Virtual packages - Provides</a> for further information.
	 * </p>
	 *
	 * @return The dependencies or an empty list, if the package doesn't have
	 *         any dependencies.
	 */
	List<PackageDependency> getProvides();


	/**
	 * <p>
	 * Returns the packages which are replaced by this package.
	 * </p>
	 * <p>
	 * See <a href=
	 * "https://www.debian.org/doc/debian-policy/ch-relationships.html#s-replaces">
	 * 7.6 Overwriting files and replacing packages - Replaces</a> for further
	 * information.
	 * </p>
	 *
	 * @return The dependencies or an empty list, if the package doesn't have
	 *         any dependencies.
	 */
	List<PackageDependency> getReplaces();


	/**
	 * <p>
	 * Returns the packages which were used to built this package.
	 * </p>
	 * <p>
	 * See <a href=
	 * "https://www.debian.org/doc/debian-policy/ch-relationships.html#s-built-using">
	 * 7.8 Additional source packages used to build the binary - Built-Using
	 * </p>
	 *
	 * @return The dependencies or an empty list, if the package doesn't have
	 *         any dependencies.
	 */
	List<PackageDependency> getBuiltUsing();


	/**
	 * <p>
	 * Returns an estimate of the total amount of disk space required to install
	 * this package.
	 * </p>
	 * <p>
	 * See <a href=
	 * "https://www.debian.org/doc/debian-policy/ch-controlfields.html#s-f-Installed-Size">
	 * </a> for further information.
	 * </p>
	 *
	 * @return The installed size.
	 */
	Size getInstalledSize();
	
	
	/**
	 * <p>
	 * Returns the description.
	 * </p>
	 *
	 * @return The description.
	 */
	Description getDescription();


	/**
	 * <p>
	 * Returns the module aliases.
	 * </p>
	 *
	 * @return The module aliases.
	 */
	String getModuleAliases();
	// TODO Create an own class / interface for the module alias.
	
	
}
