/*
 * dpkg - Debian Package library and the Debian Package Maven plugin
 * (c) Copyright 2016 Gerrit Hohl
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package net.sourceforge.javadpkg;

import java.util.List;
import java.util.Map;

/**
 * <p>
 * The copyright of a Debian package.
 * </p>
 *
 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
 * @version <b>1.0</b>, 04.05.2016 by Gerrit Hohl
 */
public interface Copyright {
	
	
	/**
	 * <p>
	 * Returns the format of the copyright.
	 * </p>
	 * <p>
	 * A format is specified by an URI of the format specification.
	 * </p>
	 * <p>
	 * Example: <blockquote>
	 * <code>http://www.debian.org/doc/packaging-manuals/copyright-format/1.0/</code>
	 * </blockquote>
	 * </p>
	 *
	 * @return The format.
	 */
	String getFormat();


	/**
	 * <p>
	 * Returns the name upstream uses for the software.
	 * </p>
	 * <p>
	 * Example: <blockquote><code>SOFTware</code></blockquote>
	 * </p>
	 *
	 * @return The name or <code>null</code>, if no name is set.
	 */
	String getUpstreamName();


	/**
	 * <p>
	 * Returns the upstream contact.
	 * </p>
	 * <p>
	 * The preferred address(es) to reach the upstream project. May be free-form
	 * text, but by convention will usually be written as a list of RFC5322
	 * addresses or URIs.
	 * </p>
	 * <p>
	 * Example: <blockquote>
	 * <code>John Doe &lt;john.doe@example.com&gt;</code></blockquote>
	 * </p>
	 *
	 * @return The contact or <code>null</code>, if no contact is set.
	 */
	String getUpstreamContact();


	/**
	 * <p>
	 * Returns the source.
	 * </p>
	 * <p>
	 * Formatted text, no synopsis: an explanation of where the upstream source
	 * came from. Typically this would be a URL, but it might be a free-form
	 * explanation.
	 * </p>
	 * <p>
	 * Example: <blockquote>
	 * <code>http://www.example.com/software/project</code></blockquote>
	 * </p>
	 *
	 * @return The source or <code>null</code>, if no source is set.
	 */
	String getSource();


	/**
	 * <p>
	 * Returns the disclaimer.
	 * </p>
	 * <p>
	 * Formatted text, no synopsis: this field is used for non-free or contrib
	 * packages to state that they are not part of Debian and to explain why.
	 * </p>
	 *
	 * @return The disclaimer or <code>null</code>, if no disclaimer is set.
	 */
	String getDisclaimer();


	/**
	 * <p>
	 * Returns the comment.
	 * </p>
	 * <p>
	 * Formatted text, no synopsis: this field can provide additional
	 * information. For example, it might quote an e-mail from upstream
	 * justifying why the license is acceptable to the main archive, or an
	 * explanation of how this version of the package has been forked from a
	 * version known to be DFSG-free, even though the current upstream version
	 * is not.
	 * </p>
	 *
	 * @return The comment or <code>null</code>, if no comment is set.
	 */
	String getComment();


	/**
	 * <p>
	 * Returns the license.
	 * </p>
	 * <p>
	 * The returned license can be reference to one of the licenses returned by
	 * the {@link #getLicenses()} method.
	 * </p>
	 *
	 * @return The license or <code>null</code>, if no license is set.
	 */
	CopyrightLicense getLicense();


	/**
	 * <p>
	 * Returns the copyright.
	 * </p>
	 * <p>
	 * Example: <blockquote><code>Copyright 2008 John Smith<br>
	 * Copyright 2009 Angela Watts</code></blockquote>
	 * </p>
	 *
	 * @return The copyright or <code>null</code>, if no copyright is set.
	 */
	String getCopyright();


	/**
	 * <p>
	 * Returns the copyrights for certain files of a Debian package.
	 * </p>
	 *
	 * @return The copyrights.
	 */
	List<FilesCopyright> getFilesCopyrights();


	/**
	 * <p>
	 * Returns the licenses referred by this copyright or one of copyrights for
	 * certain files.
	 * </p>
	 *
	 * @return The licenses.
	 * @see #getLicense()
	 * @see #getFilesCopyrights()
	 * @see FilesCopyright#getLicense()
	 */
	Map<String, CopyrightLicense> getLicenses();


}
