/*
 * dpkg - Debian Package library and the Debian Package Maven plugin
 * (c) Copyright 2016 Gerrit Hohl
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package net.sourceforge.javadpkg.store;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.List;

import org.apache.commons.compress.archivers.tar.TarArchiveEntry;
import org.apache.commons.compress.archivers.tar.TarArchiveInputStream;
import org.apache.commons.compress.archivers.tar.TarArchiveOutputStream;
import org.junit.Assert;
import org.junit.Test;

import net.sourceforge.javadpkg.AbstractDpkgTest;
import net.sourceforge.javadpkg.control.Size;
import net.sourceforge.javadpkg.io.DataSource;
import net.sourceforge.javadpkg.io.FileMetaData;
import net.sourceforge.javadpkg.io.FileMode;
import net.sourceforge.javadpkg.io.FileOwner;
import net.sourceforge.javadpkg.io.impl.FileMetaDataImpl;
import net.sourceforge.javadpkg.io.impl.FileModeImpl;
import net.sourceforge.javadpkg.io.impl.FileOwnerImpl;


/**
 * <p>
 * Performs some tests on the {@link DataStoreImpl} class.
 * </p>
 *
 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
 * @version <b>1.0</b>, 26.04.2016 by Gerrit Hohl
 * @version <b>1.1</b>, 23.01.2018 by Gerrit Hohl
 */
public class DataStoreImplTest extends AbstractDpkgTest {
	
	
	/**
	 * <p>
	 * Performs some tests on the initial state of the store.
	 * </p>
	 */
	@Test
	public void testDataStoreImpl() {
		DataStore store;
		Size size;
		ByteArrayOutputStream out;
		byte[] data;
		
		
		store = new DataStoreImpl();
		try {
			size = store.getSize();
		} catch (IOException e) {
			e.printStackTrace();
			Assert.fail("Unexpected exception: Store couldn't determine size: " + e.getMessage());
			return;
		}
		Assert.assertNotNull(size);
		Assert.assertEquals(0, size.getBytes());

		out = new ByteArrayOutputStream();
		try {
			try (TarArchiveOutputStream tarOut = new TarArchiveOutputStream(out)) {
				store.write(tarOut);
			}
		} catch (IOException e) {
			e.printStackTrace();
			Assert.fail("Unexpected exception: Store couldn't write files into archive: " + e.getMessage());
		}
		data = out.toByteArray();
		// --- An empty TAR archive has always a size of 10KB ---
		Assert.assertEquals(1536, data.length);
	}


	/**
	 * <p>
	 * Performs some tests on the {@link DataStoreImpl#addDirectory(String)}
	 * method.
	 * </p>
	 */
	@Test
	public void testAddDirectoryString() {
		DataStore store;
		ByteArrayOutputStream out;
		byte[] data;
		String content;


		store = new DataStoreImpl();
		
		
		// --- Parameters ---
		try {
			store.addDirectory(null);
			Assert.fail("Expected an exception, but none was thrown.");
		} catch (IllegalArgumentException e) {
			// --- Do nothing. Expected exception. ---
		}
		try {
			store.addDirectory("");
			Assert.fail("Expected an exception, but none was thrown.");
		} catch (IllegalArgumentException e) {
			// --- Do nothing. Expected exception. ---
		}
		try {
			store.addDirectory(".");
			Assert.fail("Expected an exception, but none was thrown.");
		} catch (IllegalArgumentException e) {
			// --- Do nothing. Expected exception. ---
		}
		try {
			store.addDirectory("./");
			Assert.fail("Expected an exception, but none was thrown.");
		} catch (IllegalArgumentException e) {
			// --- Do nothing. Expected exception. ---
		}
		
		
		// --- Add some directories ---
		store.addDirectory("/usr");
		store.addDirectory("/var/");
		store.addDirectory("./tmp/");
		try {
			store.addDirectory("/usr");
			Assert.fail("Expected an exception, but none was thrown.");
		} catch (IllegalArgumentException e) {
			// --- Do nothing. Expected exception. ---
		}
		try {
			store.addDirectory("/usr/");
			Assert.fail("Expected an exception, but none was thrown.");
		} catch (IllegalArgumentException e) {
			// --- Do nothing. Expected exception. ---
		}
		try {
			store.addDirectory("/opt/lib");
			Assert.fail("Expected an exception, but none was thrown.");
		} catch (IllegalArgumentException e) {
			// --- Do nothing. Expected exception. ---
		}
		store.addDirectory("/usr/lib");
		
		
		// --- Write the data ---
		out = new ByteArrayOutputStream();
		try {
			try (TarArchiveOutputStream tarOut = new TarArchiveOutputStream(out)) {
				store.write(tarOut);
			}
		} catch (IOException e) {
			e.printStackTrace();
			Assert.fail("Store couldn't write files into archive: " + e.getMessage());
		}
		// --- Check the data ---
		data = out.toByteArray();
		Assert.assertEquals(3584, data.length);
		try {
			content = this.readTarContent(data);
		} catch (IOException e) {
			e.printStackTrace();
			Assert.fail("Couldn't read TAR archive: " + e.getMessage());
			return;
		}
		Assert.assertEquals("drwxr-xr-x     0 root         0 root              0 /\n"
				+ "drwxr-xr-x     0 root         0 root              0 /tmp/\n"
				+ "drwxr-xr-x     0 root         0 root              0 /usr/\n"
				+ "drwxr-xr-x     0 root         0 root              0 /usr/lib/\n"
				+ "drwxr-xr-x     0 root         0 root              0 /var/\n", content);
	}
	
	
	/**
	 * <p>
	 * Performs some tests on the
	 * {@link DataStoreImpl#addDirectory(String, FileOwner, FileMode)} method.
	 * </p>
	 */
	@Test
	public void testAddDirectoryStringLongStringLongStringInt() {
		DataStore store;
		FileOwner owner;
		FileMode mode;
		ByteArrayOutputStream out;
		byte[] data;
		String content;


		store = new DataStoreImpl();
		
		
		// --- Parameters ---
		owner = new FileOwnerImpl(1234, "group", 4321, "user");
		mode = new FileModeImpl(00111);
		try {
			store.addDirectory(null, owner, mode);
			Assert.fail("Expected an exception, but none was thrown.");
		} catch (IllegalArgumentException e) {
			// --- Do nothing. Expected exception. ---
		}
		try {
			store.addDirectory("/var", null, mode);
			Assert.fail("Expected an exception, but none was thrown.");
		} catch (IllegalArgumentException e) {
			// --- Do nothing. Expected exception. ---
		}
		try {
			store.addDirectory("/var", owner, null);
			Assert.fail("Expected an exception, but none was thrown.");
		} catch (IllegalArgumentException e) {
			// --- Do nothing. Expected exception. ---
		}


		// --- Add some directories ---
		store.addDirectory("/usr", new FileOwnerImpl(1234, "group1", 4321, "user1"), new FileModeImpl(00111));
		store.addDirectory("/var/", new FileOwnerImpl(2345, "group2", 5432, "user2"), new FileModeImpl(00222));
		store.addDirectory("./tmp/", new FileOwnerImpl(3456, "group3", 6543, "user3"), new FileModeImpl(00333));
		try {
			store.addDirectory("/usr");
			Assert.fail("Expected an exception, but none was thrown.");
		} catch (IllegalArgumentException e) {
			// --- Do nothing. Expected exception. ---
		}
		try {
			store.addDirectory("/usr/");
			Assert.fail("Expected an exception, but none was thrown.");
		} catch (IllegalArgumentException e) {
			// --- Do nothing. Expected exception. ---
		}
		try {
			store.addDirectory("/opt/lib");
			Assert.fail("Expected an exception, but none was thrown.");
		} catch (IllegalArgumentException e) {
			// --- Do nothing. Expected exception. ---
		}
		store.addDirectory("/usr/lib", new FileOwnerImpl(4567, "group4", 7654, "user4"), new FileModeImpl(00444));
		
		
		// --- Write the data ---
		out = new ByteArrayOutputStream();
		try {
			try (TarArchiveOutputStream tarOut = new TarArchiveOutputStream(out)) {
				store.write(tarOut);
			}
		} catch (IOException e) {
			e.printStackTrace();
			Assert.fail("Store couldn't write files into archive: " + e.getMessage());
		}
		// --- Check the data ---
		data = out.toByteArray();
		Assert.assertEquals(3584, data.length);
		try {
			content = this.readTarContent(data);
		} catch (IOException e) {
			e.printStackTrace();
			Assert.fail("Couldn't read TAR archive: " + e.getMessage());
			return;
		}
		Assert.assertEquals("drwxr-xr-x     0 root         0 root              0 /\n"
				+ "d-wx-wx-wx  3456 group3    6543 user3             0 /tmp/\n"
				+ "d--x--x--x  1234 group1    4321 user1             0 /usr/\n"
				+ "dr--r--r--  4567 group4    7654 user4             0 /usr/lib/\n"
				+ "d-w--w--w-  2345 group2    5432 user2             0 /var/\n", content);
	}
	
	
	/**
	 * <p>
	 * Performs some tests on the
	 * {@link DataStoreImpl#addFile(DataSource, String)} method.
	 * </p>
	 */
	@SuppressWarnings("resource")
	@Test
	public void testAddFileDataSourceString() {
		DataStore store;
		DataSource source;
		ByteArrayOutputStream out;
		byte[] data;
		String content;


		store = new DataStoreImpl();
		
		
		// --- Parameters ---
		source = AbstractDpkgTest.createSource(524288, "file1");
		try {
			store.addFile(null, "/hello.txt");
			Assert.fail("Expected an exception, but none was thrown.");
		} catch (IllegalArgumentException e) {
			// --- Do nothing. Expected exception. ---
		}
		try {
			store.addFile(source, null);
			Assert.fail("Expected an exception, but none was thrown.");
		} catch (IllegalArgumentException e) {
			// --- Do nothing. Expected exception. ---
		}
		try {
			store.addFile(source, "");
			Assert.fail("Expected an exception, but none was thrown.");
		} catch (IllegalArgumentException e) {
			// --- Do nothing. Expected exception. ---
		}
		try {
			store.addFile(source, ".");
			Assert.fail("Expected an exception, but none was thrown.");
		} catch (IllegalArgumentException e) {
			// --- Do nothing. Expected exception. ---
		}
		try {
			store.addFile(source, "./");
			Assert.fail("Expected an exception, but none was thrown.");
		} catch (IllegalArgumentException e) {
			// --- Do nothing. Expected exception. ---
		}
		try {
			store.addFile(source, "./");
			Assert.fail("Expected an exception, but none was thrown.");
		} catch (IllegalArgumentException e) {
			// --- Do nothing. Expected exception. ---
		}


		// --- Add some files ---
		store.addFile(source, "/hello.txt");
		source = AbstractDpkgTest.createSource(1048576, "file2");
		try {
			store.addFile(source, "/hello.txt");
			Assert.fail("Expected an exception, but none was thrown.");
		} catch (IllegalArgumentException e) {
			// --- Do nothing. Expected exception. ---
		}
		try {
			store.addFile(source, "/var/hello.txt");
		} catch (IllegalArgumentException e) {
			// --- Do nothing. Expected exception. ---
		}
		store.addDirectory("/var");
		store.addFile(source, "/var/hello.txt");


		// --- Write the data ---
		out = new ByteArrayOutputStream();
		try {
			try (TarArchiveOutputStream tarOut = new TarArchiveOutputStream(out)) {
				store.write(tarOut);
			}
		} catch (IOException e) {
			e.printStackTrace();
			Assert.fail("Store couldn't write files into archive: " + e.getMessage());
		}
		// --- Check the data ---
		data = out.toByteArray();
		Assert.assertEquals(1571840 + 4096, data.length);
		try {
			content = this.readTarContent(data);
		} catch (IOException e) {
			e.printStackTrace();
			Assert.fail("Couldn't read TAR archive: " + e.getMessage());
			return;
		}
		Assert.assertEquals("drwxr-xr-x     0 root         0 root              0 /\n"
				+ "drwxr-xr-x     0 root         0 root              0 /var/\n"
				+ "-rw-r--r--     0 root         0 root        1048576 /var/hello.txt\n"
				+ "-rw-r--r--     0 root         0 root         524288 /hello.txt\n", content);
	}
	
	
	/**
	 * <p>
	 * Performs some tests on the
	 * {@link DataStoreImpl#addFile(DataSource, String, FileOwner, FileMode)}
	 * method.
	 * </p>
	 */
	@SuppressWarnings("resource")
	@Test
	public void testAddFileDataSourceStringLongStringLongStringInt() {
		DataStore store;
		DataSource source;
		FileOwner owner;
		FileMode mode;
		ByteArrayOutputStream out;
		byte[] data;
		String content;


		store = new DataStoreImpl();


		// --- Parameters ---
		source = AbstractDpkgTest.createSource(524288, "file1");
		owner = new FileOwnerImpl(1234, "group1", 4321, "user1");
		mode = new FileModeImpl(00111);
		try {
			store.addFile(null, "/hello.txt", owner, mode);
			Assert.fail("Expected an exception, but none was thrown.");
		} catch (IllegalArgumentException e) {
			// --- Do nothing. Expected exception. ---
		}
		try {
			store.addFile(source, null, owner, mode);
			Assert.fail("Expected an exception, but none was thrown.");
		} catch (IllegalArgumentException e) {
			// --- Do nothing. Expected exception. ---
		}
		try {
			store.addFile(source, "/hello.txt", null, mode);
			Assert.fail("Expected an exception, but none was thrown.");
		} catch (IllegalArgumentException e) {
			// --- Do nothing. Expected exception. ---
		}
		try {
			store.addFile(source, "/hello.txt", owner, null);
			Assert.fail("Expected an exception, but none was thrown.");
		} catch (IllegalArgumentException e) {
			// --- Do nothing. Expected exception. ---
		}


		// --- Add some files ---
		try {
			store.addFile(source, "", owner, mode);
			Assert.fail("Expected an exception, but none was thrown.");
		} catch (IllegalArgumentException e) {
			// --- Do nothing. Expected exception. ---
		}
		try {
			store.addFile(source, ".", owner, mode);
			Assert.fail("Expected an exception, but none was thrown.");
		} catch (IllegalArgumentException e) {
			// --- Do nothing. Expected exception. ---
		}
		try {
			store.addFile(source, "./", owner, mode);
			Assert.fail("Expected an exception, but none was thrown.");
		} catch (IllegalArgumentException e) {
			// --- Do nothing. Expected exception. ---
		}
		try {
			store.addFile(source, "./", owner, mode);
			Assert.fail("Expected an exception, but none was thrown.");
		} catch (IllegalArgumentException e) {
			// --- Do nothing. Expected exception. ---
		}
		store.addFile(source, "/hello.txt", owner, mode);
		source = AbstractDpkgTest.createSource(1048576, "file2");
		try {
			store.addFile(source, "/hello.txt", new FileOwnerImpl(2345, "group2", 5432, "user2"), new FileModeImpl(00222));
			Assert.fail("Expected an exception, but none was thrown.");
		} catch (IllegalArgumentException e) {
			// --- Do nothing. Expected exception. ---
		}
		try {
			store.addFile(source, "/var/hello.txt", new FileOwnerImpl(2345, "group2", 5432, "user2"), new FileModeImpl(00222));
		} catch (IllegalArgumentException e) {
			// --- Do nothing. Expected exception. ---
		}
		store.addDirectory("/var");
		store.addFile(source, "/var/hello.txt", new FileOwnerImpl(2345, "group2", 5432, "user2"), new FileModeImpl(00222));


		// --- Write the data ---
		out = new ByteArrayOutputStream();
		try {
			try (TarArchiveOutputStream tarOut = new TarArchiveOutputStream(out)) {
				store.write(tarOut);
			}
		} catch (IOException e) {
			e.printStackTrace();
			Assert.fail("Store couldn't write files into archive: " + e.getMessage());
		}
		// --- Check the data ---
		data = out.toByteArray();
		Assert.assertEquals(1571840 + 4096, data.length);
		try {
			content = this.readTarContent(data);
		} catch (IOException e) {
			e.printStackTrace();
			Assert.fail("Couldn't read TAR archive: " + e.getMessage());
			return;
		}
		Assert.assertEquals("drwxr-xr-x     0 root         0 root              0 /\n"
				+ "drwxr-xr-x     0 root         0 root              0 /var/\n"
				+ "--w--w--w-  2345 group2    5432 user2       1048576 /var/hello.txt\n"
				+ "---x--x--x  1234 group1    4321 user1        524288 /hello.txt\n", content);
	}
	
	
	/**
	 * <p>
	 * Performs some tests on the
	 * {@link DataStoreImpl#addSymLink(String, String, FileOwner, FileMode)}
	 * method.
	 * </p>
	 */
	@Test
	public void testAddSymLink() {
		DataStore store;
		FileOwner owner;
		FileMode mode;
		ByteArrayOutputStream out;
		byte[] data;
		String content;


		store = new DataStoreImpl();
		owner = new FileOwnerImpl(1234, "group1", 4321, "user1");
		mode = new FileModeImpl(00111);
		
		
		// --- Parameters ---
		try {
			store.addSymLink(null, "/usr/local/mypackage/myprogram", owner, mode);
			Assert.fail("Expected an exception, but none was thrown.");
		} catch (IllegalArgumentException e) {
			// --- Do nothing. Expected exception. ---
		}
		try {
			store.addSymLink("/link", null, owner, mode);
			Assert.fail("Expected an exception, but none was thrown.");
		} catch (IllegalArgumentException e) {
			// --- Do nothing. Expected exception. ---
		}
		try {
			store.addSymLink("/link", "/usr/local/mypackage/myprogram", null, mode);
			Assert.fail("Expected an exception, but none was thrown.");
		} catch (IllegalArgumentException e) {
			// --- Do nothing. Expected exception. ---
		}
		try {
			store.addSymLink("/link", "/usr/local/mypackage/myprogram", owner, null);
			Assert.fail("Expected an exception, but none was thrown.");
		} catch (IllegalArgumentException e) {
			// --- Do nothing. Expected exception. ---
		}
		
		
		// --- Add a link ---
		store.addSymLink("/link", "/usr/local/mypackage/myprogram", owner, mode);


		// --- Write the data ---
		out = new ByteArrayOutputStream();
		try {
			try (TarArchiveOutputStream tarOut = new TarArchiveOutputStream(out)) {
				store.write(tarOut);
			}
		} catch (IOException e) {
			e.printStackTrace();
			Assert.fail("Store couldn't write files into archive: " + e.getMessage());
		}
		// --- Check the data ---
		data = out.toByteArray();
		Assert.assertEquals(2048, data.length);
		try {
			content = this.readTarContent(data);
		} catch (IOException e) {
			e.printStackTrace();
			Assert.fail("Couldn't read TAR archive: " + e.getMessage());
			return;
		}
		Assert.assertEquals("drwxr-xr-x     0 root         0 root              0 /\n"
				+ "l--x--x--x  1234 group1    4321 user1            -1 /link\n", content);
	}
	
	
	/**
	 * <p>
	 * Performs some tests on the {@link DataStoreImpl#exists(String)} method.
	 * </p>
	 */
	@SuppressWarnings("resource")
	@Test
	public void testExists() {
		DataStore store;
		DataSource source;


		store = new DataStoreImpl();
		
		
		// --- Parameters ---
		try {
			store.exists(null);
			Assert.fail("Expected an exception, but none was thrown.");
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		}


		// --- Not existing ---
		Assert.assertFalse(store.exists("/usr/local/myprogram"));
		Assert.assertFalse(store.exists("/usr/local/myprogram/program"));


		// --- Existing ---
		source = AbstractDpkgTest.createSource(524288, "file1");
		store.addDirectory("/usr");
		store.addDirectory("/usr/local");
		store.addDirectory("/usr/local/myprogram");
		store.addFile(source, "/usr/local/myprogram/program");
	}


	/**
	 * <p>
	 * Performs some tests on the {@link DataStoreImpl#getSize()} method.
	 * </p>
	 */
	@SuppressWarnings("resource")
	@Test
	public void testGetSize() {
		DataStore store;
		Size size;
		DataSource source;
		
		
		// --- Empty store ---
		store = new DataStoreImpl();
		try {
			size = store.getSize();
		} catch (IOException e) {
			e.printStackTrace();
			Assert.fail("Unexpected exception: Store couldn't determine size: " + e.getMessage());
			return;
		}
		Assert.assertNotNull(size);
		Assert.assertEquals(0, size.getBytes());


		// --- Store with some directories, but no file ---
		store.addDirectory("/var");
		store.addDirectory("/usr");
		try {
			size = store.getSize();
		} catch (IOException e) {
			e.printStackTrace();
			Assert.fail("Unexpected exception: Store couldn't determine size: " + e.getMessage());
			return;
		}
		Assert.assertNotNull(size);
		Assert.assertEquals(0, size.getBytes());
		
		
		// --- Store with some directories and some files ---
		source = AbstractDpkgTest.createSource(524288, "file1");
		store.addFile(source, "/var/file1");
		source = AbstractDpkgTest.createSource(1048576, "file2");
		store.addFile(source, "/var/file2");
		try {
			size = store.getSize();
		} catch (IOException e) {
			e.printStackTrace();
			Assert.fail("Unexpected exception: Store couldn't determine size: " + e.getMessage());
			return;
		}
		Assert.assertNotNull(size);
		Assert.assertEquals(1572864, size.getBytes());
	}
	
	
	/**
	 * <p>
	 * Performs some tests on the
	 * {@link DataStoreImpl#createFileHashes(MessageDigest)} method.
	 * </p>
	 */
	@SuppressWarnings("resource")
	@Test
	public void testCreateMD5FileHashes() {
		MessageDigest digest;
		DataStore store;
		DataSource source;
		List<FileHash> fileHashes;
		FileHash fileHash;
		
		
		try {
			digest = MessageDigest.getInstance("MD5");
		} catch (NoSuchAlgorithmException e) {
			e.printStackTrace();
			Assert.fail(e.getMessage());
			return;
		}

		// --- Empty store ---
		store = new DataStoreImpl();
		try {
			fileHashes = store.createFileHashes(digest);
		} catch (IOException e) {
			e.printStackTrace();
			Assert.fail("Couldn't create file hashes: " + e.getMessage());
			return;
		}
		Assert.assertNotNull(fileHashes);
		Assert.assertEquals(0, fileHashes.size());
		
		
		// --- Populated store ---
		source = AbstractDpkgTest.createSource(524288, "file1");
		store.addFile(source, "/file1");
		source = AbstractDpkgTest.createSource(1048576, "file2");
		store.addFile(source, "/file2");
		try {
			fileHashes = store.createFileHashes(digest);
		} catch (IOException e) {
			e.printStackTrace();
			Assert.fail("Couldn't create file hashes: " + e.getMessage());
			return;
		}
		Assert.assertNotNull(fileHashes);
		Assert.assertEquals(2, fileHashes.size());

		fileHash = fileHashes.get(0);
		Assert.assertEquals("file1", fileHash.getName());
		Assert.assertEquals("/file1", fileHash.getPath());
		Assert.assertArrayEquals(new byte[] { 0x59, 0x07, 0x15, (byte) 0x90, 0x09, (byte) 0x9D, 0x21, (byte) 0xDD, 0x43,
				(byte) 0x98, (byte) 0x96, 0x59, 0x23, 0x38, (byte) 0xBF, (byte) 0x95 }, fileHash.getHash());
		Assert.assertEquals("59071590099d21dd439896592338bf95", fileHash.getHashAsHex());

		fileHash = fileHashes.get(1);
		Assert.assertEquals("file2", fileHash.getName());
		Assert.assertEquals("/file2", fileHash.getPath());
		Assert.assertArrayEquals(new byte[] { (byte) 0xB6, (byte) 0xD8, 0x1B, 0x36, 0x0A, 0x56, 0x72, (byte) 0xD8, 0x0C, 0x27,
				0x43, 0x0F, 0x39, 0x15, 0x3E, 0x2C }, fileHash.getHash());
		Assert.assertEquals("b6d81b360a5672d80c27430f39153e2c", fileHash.getHashAsHex());
	}
	
	
	/**
	 * <p>
	 * Reads the content of the TAR archive.
	 * </p>
	 *
	 * @param data
	 *            The data of the TAR archive.
	 * @return The content (folder structure).
	 * @throws IOException
	 *             If an I/O error occurs.
	 */
	private String readTarContent(byte[] data) throws IOException {
		StringBuilder sb;
		ByteArrayInputStream in;
		TarArchiveEntry entry;
		FileOwner owner;
		FileMode mode;
		FileMetaData file;
		Long groupId, userId, length;


		in = new ByteArrayInputStream(data);
		sb = new StringBuilder();
		try (TarArchiveInputStream tarIn = new TarArchiveInputStream(in)) {
			while ((entry = tarIn.getNextTarEntry()) != null) {
				// --- Get file meta data ---
				owner = new FileOwnerImpl(entry.getLongGroupId(), entry.getGroupName(), entry.getLongUserId(),
						entry.getUserName());
				mode = new FileModeImpl(entry.getMode());
				if (entry.isDirectory()) {
					file = FileMetaDataImpl.createDirectoryMetaData(entry.getName(), owner, mode, entry.getLastModifiedDate());
				} else if (entry.isSymbolicLink()) {
					file = FileMetaDataImpl.createSymbolicLinkMetaData(entry.getName(), entry.getLinkName(), owner, mode,
							entry.getLastModifiedDate());
				} else if (entry.isFile()) {
					file = FileMetaDataImpl.createFileMetaData(entry.getName(), owner, mode, entry.getSize(),
							entry.getLastModifiedDate());
				} else
					throw new IOException("Entry |" + entry.getName() + "| is not a directory, symbolic link or file.");
				
				// --- Prepare propertoes ---
				groupId = Long.valueOf(file.getOwner().getGroupId());
				userId = Long.valueOf(file.getOwner().getUserId());
				if (file.isDirectory() && (file.getLength() < 0)) {
					length = Long.valueOf(0);
				} else {
					length = Long.valueOf(file.getLength());
				}

				// --- Add properties ---
				sb.append(String.format("%s %5d %-8s %5d %-8s %10d %s\n", file.getModeAsText(), groupId,
						file.getOwner().getGroupName(), userId, file.getOwner().getUserName(), length, file.getAbsolutePath()));
			}
		}
		return sb.toString();
	}


}
