/*
 * dpkg - Debian Package library and the Debian Package Maven plugin
 * (c) Copyright 2016 Gerrit Hohl
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package net.sourceforge.javadpkg.replace;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.junit.Assert;
import org.junit.Test;

import net.sourceforge.javadpkg.AbstractDpkgTest;
import net.sourceforge.javadpkg.Context;
import net.sourceforge.javadpkg.Warning;
import net.sourceforge.javadpkg.impl.ContextImpl;
import net.sourceforge.javadpkg.replace.ReplacementException;
import net.sourceforge.javadpkg.replace.Replacements;
import net.sourceforge.javadpkg.replace.ReplacementsMap;
import net.sourceforge.javadpkg.replace.Replacer;
import net.sourceforge.javadpkg.replace.ReplacerImpl;


/**
 * <p>
 * Performs some tests on the {@link ReplacerImpl} class.
 * </p>
 *
 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
 * @version <b>1.0</b>, 02.05.2016 by Gerrit Hohl
 */
public class ReplacerImplTest extends AbstractDpkgTest {
	
	
	/**
	 * <p>
	 * Performs some tests on the {@link ReplacerImpl} class testing the
	 * parameters.
	 * </p>
	 */
	@Test
	public void testParameters() {
		Replacer replacer;
		Replacements replacements;
		Context context;


		replacer = new ReplacerImpl();


		// --- Parameters ---
		replacements = new ReplacementsMap();
		context = new ContextImpl();
		try {
			replacer.replace(null, replacements, context);
			Assert.fail("Expected an exception, but none was thrown.");
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		} catch (ReplacementException e) {
			e.printStackTrace();
			Assert.fail("An unexpected exception was thrown: " + e.getMessage());
			return;
		}
		try {
			replacer.replace("THISVERSION=\"${deb.version}\"", null, context);
			Assert.fail("Expected an exception, but none was thrown.");
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		} catch (ReplacementException e) {
			e.printStackTrace();
			Assert.fail("An unexpected exception was thrown: " + e.getMessage());
			return;
		}
		try {
			replacer.replace("THISVERSION=\"${deb.version}\"", replacements, null);
			Assert.fail("Expected an exception, but none was thrown.");
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		} catch (ReplacementException e) {
			e.printStackTrace();
			Assert.fail("An unexpected exception was thrown: " + e.getMessage());
			return;
		}
	}


	/**
	 * <p>
	 * Performs some tests on the {@link ReplacerImpl} class producing some
	 * successful replacements.
	 * </p>
	 */
	@Test
	public void testReplaceSuccessful() {
		Replacer replacer;
		Map<String, String> variables;
		Replacements replacements;
		Context context;
		String result;
		List<Warning> warnings;


		replacer = new ReplacerImpl();


		// --- Replace successfully ---
		context = new ContextImpl();
		variables = new HashMap<>();
		variables.put("deb.version", "1.0.0");
		replacements = new ReplacementsMap(variables);
		try {
			result = replacer.replace("THISVERSION=\"${deb.version}\"", replacements, context);
		} catch (ReplacementException e) {
			e.printStackTrace();
			Assert.fail("An unexpected exception was thrown: " + e.getMessage());
			return;
		}
		warnings = context.getWarnings();
		Assert.assertNotNull(warnings);
		Assert.assertEquals(0, warnings.size());
		Assert.assertNotNull(result);
		Assert.assertEquals("THISVERSION=\"1.0.0\"", result);


		context = new ContextImpl();
		variables = new HashMap<>();
		variables.put("link", "/opt/mainfolder");
		variables.put("folder", "/opt/versionfolder-1.0.0");
		replacements = new ReplacementsMap(variables);
		try {
			result = replacer.replace("        ln --force --symbolic ${folder} ${link}", replacements, context);
		} catch (ReplacementException e) {
			e.printStackTrace();
			Assert.fail("An unexpected exception was thrown: " + e.getMessage());
			return;
		}
		warnings = context.getWarnings();
		Assert.assertNotNull(warnings);
		Assert.assertEquals(0, warnings.size());
		Assert.assertNotNull(result);
		Assert.assertEquals("        ln --force --symbolic /opt/versionfolder-1.0.0 /opt/mainfolder", result);
	}
	
	
	/**
	 * <p>
	 * Performs some tests on the {@link ReplacerImpl} class producing warnings
	 * </p>
	 */
	@Test
	public void testReplaceWarning() {
		Replacer replacer;
		Replacements replacements;
		Context context;
		String result;
		List<Warning> warnings;
		Warning warning;


		replacer = new ReplacerImpl();
		
		
		// --- Warnings ---
		context = new ContextImpl();
		replacements = new ReplacementsMap();
		try {
			result = replacer.replace("THISVERSION=\"${deb.version}\"", replacements, context);
		} catch (ReplacementException e) {
			e.printStackTrace();
			Assert.fail("An unexpected exception was thrown: " + e.getMessage());
			return;
		}
		warnings = context.getWarnings();
		Assert.assertNotNull(warnings);
		Assert.assertEquals(1, warnings.size());
		warning = warnings.get(0);
		Assert.assertNotNull(warning);
		Assert.assertEquals("Found unknown variable |deb.version|. The variable will be ignored and not replaced.",
				warning.getText());
		Assert.assertNotNull(result);
		Assert.assertEquals("THISVERSION=\"${deb.version}\"", result);
	}
	
	
	/**
	 * <p>
	 * Performs some tests on the {@link ReplacerImpl} class producing errors
	 * </p>
	 */
	@Test
	public void testReplaceError() {
		Replacer replacer;
		Replacements replacements;
		Context context;


		replacer = new ReplacerImpl();
		
		
		// --- Error ---
		context = new ContextImpl();
		replacements = new ReplacementsMap();
		try {
			replacer.replace("THISVERSION=\"${deb.version\"", replacements, context);
			Assert.fail("Expected an exception, but none was thrown.");
		} catch (ReplacementException e) {
			// --- Expected exception. Do nothing. ---
		}
	}
	
	
}
