/*
 * dpkg - Debian Package library and the Debian Package Maven plugin
 * (c) Copyright 2016 Gerrit Hohl
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package net.sourceforge.javadpkg.io.impl;

import java.util.Date;

import org.junit.Assert;
import org.junit.Test;

import net.sourceforge.javadpkg.AbstractDpkgTest;
import net.sourceforge.javadpkg.io.FileMetaData;
import net.sourceforge.javadpkg.io.FileMode;
import net.sourceforge.javadpkg.io.FileOwner;
import net.sourceforge.javadpkg.io.impl.FileMetaDataImpl;
import net.sourceforge.javadpkg.io.impl.FileModeImpl;
import net.sourceforge.javadpkg.io.impl.FileOwnerImpl;


/**
 * <p>
 * Performs some tests on the {@link FileMetaDataImpl} class.
 * </p>
 *
 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
 * @version <b>1.0</b>, 12.05.2016 by Gerrit Hohl
 */
public class FileMetaDataImplTest extends AbstractDpkgTest {
	
	
	/**
	 * <p>
	 * Performs some tests on the
	 * {@link FileMetaDataImpl#FileMetaDataImpl(String, String, FileOwner, FileMode, Date)}
	 * method.
	 * </p>
	 */
	@Test
	public void testFileMetaDataImplStringStringFileOwnerFileModeDate() {
		FileMetaData file;
		FileOwner owner;
		FileMode mode;
		Date lastModifiedDate;
		
		
		// --- Parameters ---
		owner = new FileOwnerImpl(0, "root", 0, "root");
		mode = new FileModeImpl(00755);
		lastModifiedDate = new Date(1463043896603L);
		try {
			file = new FileMetaDataImpl((String) null, "usr", owner, mode, lastModifiedDate);
			Assert.fail("Expected an exception, but none was thrown.");
			return;
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		}
		try {
			file = new FileMetaDataImpl("", "usr", owner, mode, lastModifiedDate);
			Assert.fail("Expected an exception, but none was thrown.");
			return;
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		}
		try {
			file = new FileMetaDataImpl("/", null, owner, mode, lastModifiedDate);
			Assert.fail("Expected an exception, but none was thrown.");
			return;
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		}
		try {
			file = new FileMetaDataImpl("/", "/", owner, mode, lastModifiedDate);
			Assert.fail("Expected an exception, but none was thrown.");
			return;
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		}
		try {
			file = new FileMetaDataImpl("/", "", null, mode, lastModifiedDate);
			Assert.fail("Expected an exception, but none was thrown.");
			return;
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		}
		try {
			file = new FileMetaDataImpl("/", "usr", owner, null, lastModifiedDate);
			Assert.fail("Expected an exception, but none was thrown.");
			return;
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		}
		try {
			file = new FileMetaDataImpl("/", "usr", owner, mode, null);
			Assert.fail("Expected an exception, but none was thrown.");
			return;
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		}
		
		
		// --- Root ---
		file = new FileMetaDataImpl("/", "", owner, mode, lastModifiedDate);
		Assert.assertEquals("/", file.getPath());
		Assert.assertEquals("", file.getName());
		Assert.assertEquals("/", file.getAbsolutePath());
		Assert.assertFalse(file.isFile());
		Assert.assertTrue(file.isDirectory());
		Assert.assertFalse(file.isSymbolicLink());
		Assert.assertNull(file.getTargetPath());
		Assert.assertSame(owner, file.getOwner());
		Assert.assertSame(mode, file.getMode());
		Assert.assertEquals("drwxr-xr-x", file.getModeAsText());
		Assert.assertEquals(-1, file.getLength());
		Assert.assertSame(lastModifiedDate, file.getLastModifiedDate());
		
		
		// --- Directory /usr ---
		file = new FileMetaDataImpl("./", "usr", owner, mode, lastModifiedDate);
		Assert.assertEquals("/", file.getPath());
		Assert.assertEquals("usr", file.getName());
		Assert.assertEquals("/usr/", file.getAbsolutePath());
		Assert.assertFalse(file.isFile());
		Assert.assertTrue(file.isDirectory());
		Assert.assertFalse(file.isSymbolicLink());
		Assert.assertNull(file.getTargetPath());
		Assert.assertSame(owner, file.getOwner());
		Assert.assertSame(mode, file.getMode());
		Assert.assertEquals("drwxr-xr-x", file.getModeAsText());
		Assert.assertEquals(-1, file.getLength());
		Assert.assertSame(lastModifiedDate, file.getLastModifiedDate());
		
		
		// --- Directory /usr/local ---
		file = new FileMetaDataImpl("usr", "local", owner, mode, lastModifiedDate);
		Assert.assertEquals("/usr/", file.getPath());
		Assert.assertEquals("local", file.getName());
		Assert.assertEquals("/usr/local/", file.getAbsolutePath());
		Assert.assertFalse(file.isFile());
		Assert.assertTrue(file.isDirectory());
		Assert.assertFalse(file.isSymbolicLink());
		Assert.assertNull(file.getTargetPath());
		Assert.assertSame(owner, file.getOwner());
		Assert.assertSame(mode, file.getMode());
		Assert.assertEquals("drwxr-xr-x", file.getModeAsText());
		Assert.assertEquals(-1, file.getLength());
		Assert.assertSame(lastModifiedDate, file.getLastModifiedDate());
	}
	
	
	/**
	 * <p>
	 * Performs some tests on the
	 * {@link FileMetaDataImpl#FileMetaDataImpl(FileMetaData, String, FileOwner, FileMode, Date)}
	 * method.
	 * </p>
	 */
	@Test
	public void testFileMetaDataImplFileMetaDataStringFileOwnerFileModeDate() {
		FileMetaData file, parent;
		FileOwner owner;
		FileMode mode;
		Date lastModifiedDate;
		
		
		// --- Parameters ---
		owner = new FileOwnerImpl(0, "root", 0, "root");
		mode = new FileModeImpl(00755);
		lastModifiedDate = new Date(1463043896603L);
		parent = new FileMetaDataImpl("/", "", new FileOwnerImpl(0, "root", 0, "root"), new FileModeImpl(00755),
				new Date(1463043896603L));
		try {
			file = new FileMetaDataImpl((FileMetaData) null, "usr", owner, mode, lastModifiedDate);
			Assert.fail("Expected an exception, but none was thrown.");
			return;
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		}
		try {
			file = new FileMetaDataImpl(parent, null, owner, mode, lastModifiedDate);
			Assert.fail("Expected an exception, but none was thrown.");
			return;
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		}
		try {
			file = new FileMetaDataImpl(parent, "/", owner, mode, lastModifiedDate);
			Assert.fail("Expected an exception, but none was thrown.");
			return;
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		}
		try {
			file = new FileMetaDataImpl(parent, "usr", null, mode, lastModifiedDate);
			Assert.fail("Expected an exception, but none was thrown.");
			return;
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		}
		try {
			file = new FileMetaDataImpl(parent, "usr", owner, null, lastModifiedDate);
			Assert.fail("Expected an exception, but none was thrown.");
			return;
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		}
		try {
			file = new FileMetaDataImpl(parent, "usr", owner, mode, null);
			Assert.fail("Expected an exception, but none was thrown.");
			return;
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		}


		// --- Directory /usr ---
		file = new FileMetaDataImpl(parent, "usr", owner, mode, lastModifiedDate);
		Assert.assertEquals("/", file.getPath());
		Assert.assertEquals("usr", file.getName());
		Assert.assertEquals("/usr/", file.getAbsolutePath());
		Assert.assertFalse(file.isFile());
		Assert.assertTrue(file.isDirectory());
		Assert.assertFalse(file.isSymbolicLink());
		Assert.assertNull(file.getTargetPath());
		Assert.assertSame(owner, file.getOwner());
		Assert.assertSame(mode, file.getMode());
		Assert.assertEquals("drwxr-xr-x", file.getModeAsText());
		Assert.assertEquals(-1, file.getLength());
		Assert.assertSame(lastModifiedDate, file.getLastModifiedDate());
		
		
		// --- Directory /usr/local ---
		parent = file;
		file = new FileMetaDataImpl(parent, "local", owner, mode, lastModifiedDate);
		Assert.assertEquals("/usr/", file.getPath());
		Assert.assertEquals("local", file.getName());
		Assert.assertEquals("/usr/local/", file.getAbsolutePath());
		Assert.assertFalse(file.isFile());
		Assert.assertTrue(file.isDirectory());
		Assert.assertFalse(file.isSymbolicLink());
		Assert.assertNull(file.getTargetPath());
		Assert.assertSame(owner, file.getOwner());
		Assert.assertSame(mode, file.getMode());
		Assert.assertEquals("drwxr-xr-x", file.getModeAsText());
		Assert.assertEquals(-1, file.getLength());
		Assert.assertSame(lastModifiedDate, file.getLastModifiedDate());
	}


	/**
	 * <p>
	 * Performs some tests on the
	 * {@link FileMetaDataImpl#FileMetaDataImpl(String, String, FileOwner, FileMode, long, Date)}
	 * method.
	 * </p>
	 */
	@Test
	public void testFileMetaDataImplStringStringFileOwnerFileModeLongDate() {
		FileMetaData file;
		FileOwner owner;
		FileMode mode;
		Date lastModifiedDate;
		
		
		// --- Parameters ---
		owner = new FileOwnerImpl(0, "root", 0, "root");
		mode = new FileModeImpl(00755);
		lastModifiedDate = new Date(1463043896603L);
		try {
			file = new FileMetaDataImpl((String) null, "usr", owner, mode, 0, lastModifiedDate);
			Assert.fail("Expected an exception, but none was thrown.");
			return;
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		}
		try {
			file = new FileMetaDataImpl("", "usr", owner, mode, 0, lastModifiedDate);
			Assert.fail("Expected an exception, but none was thrown.");
			return;
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		}
		try {
			file = new FileMetaDataImpl("/", null, owner, mode, 0, lastModifiedDate);
			Assert.fail("Expected an exception, but none was thrown.");
			return;
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		}
		try {
			file = new FileMetaDataImpl("/", "/", owner, mode, 0, lastModifiedDate);
			Assert.fail("Expected an exception, but none was thrown.");
			return;
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		}
		try {
			file = new FileMetaDataImpl("/", "usr", null, mode, 0, lastModifiedDate);
			Assert.fail("Expected an exception, but none was thrown.");
			return;
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		}
		try {
			file = new FileMetaDataImpl("/", "usr", owner, null, 0, lastModifiedDate);
			Assert.fail("Expected an exception, but none was thrown.");
			return;
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		}
		try {
			file = new FileMetaDataImpl("/", "usr", owner, null, -2, lastModifiedDate);
			Assert.fail("Expected an exception, but none was thrown.");
			return;
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		}
		try {
			file = new FileMetaDataImpl("/", "usr", owner, mode, 0, null);
			Assert.fail("Expected an exception, but none was thrown.");
			return;
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		}


		// --- Executable file ---
		file = new FileMetaDataImpl("./usr/local/mypackage", "myprogram", owner, mode, 12345, lastModifiedDate);
		Assert.assertEquals("/usr/local/mypackage/", file.getPath());
		Assert.assertEquals("myprogram", file.getName());
		Assert.assertEquals("/usr/local/mypackage/myprogram", file.getAbsolutePath());
		Assert.assertTrue(file.isFile());
		Assert.assertFalse(file.isDirectory());
		Assert.assertFalse(file.isSymbolicLink());
		Assert.assertNull(file.getTargetPath());
		Assert.assertSame(owner, file.getOwner());
		Assert.assertSame(mode, file.getMode());
		Assert.assertEquals("-rwxr-xr-x", file.getModeAsText());
		Assert.assertEquals(12345, file.getLength());
		Assert.assertSame(lastModifiedDate, file.getLastModifiedDate());


		// --- Normal file ---
		mode = new FileModeImpl(0644);
		file = new FileMetaDataImpl("usr/local/mypackage/", "myprogram", owner, mode, 12345, lastModifiedDate);
		Assert.assertEquals("/usr/local/mypackage/", file.getPath());
		Assert.assertEquals("myprogram", file.getName());
		Assert.assertEquals("/usr/local/mypackage/myprogram", file.getAbsolutePath());
		Assert.assertTrue(file.isFile());
		Assert.assertFalse(file.isDirectory());
		Assert.assertFalse(file.isSymbolicLink());
		Assert.assertNull(file.getTargetPath());
		Assert.assertSame(owner, file.getOwner());
		Assert.assertSame(mode, file.getMode());
		Assert.assertEquals("-rw-r--r--", file.getModeAsText());
		Assert.assertEquals(12345, file.getLength());
		Assert.assertSame(lastModifiedDate, file.getLastModifiedDate());
	}


	/**
	 * <p>
	 * Performs some tests on the
	 * {@link FileMetaDataImpl#FileMetaDataImpl(FileMetaData, String, FileOwner, FileMode, long, Date)}
	 * method.
	 * </p>
	 */
	@Test
	public void testFileMetaDataImplFileMetaDataStringFileOwnerFileModeLongDate() {
		FileMetaData file, parent;
		FileOwner owner;
		FileMode mode;
		Date lastModifiedDate;
		
		
		// --- Parameters ---
		owner = new FileOwnerImpl(0, "root", 0, "root");
		mode = new FileModeImpl(00755);
		lastModifiedDate = new Date(1463043896603L);
		parent = new FileMetaDataImpl("/usr/local/", "mypackage", new FileOwnerImpl(0, "root", 0, "root"),
				new FileModeImpl(00755), new Date(1463043896603L));
		try {
			file = new FileMetaDataImpl((FileMetaData) null, "usr", owner, mode, 0, lastModifiedDate);
			Assert.fail("Expected an exception, but none was thrown.");
			return;
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		}
		try {
			file = new FileMetaDataImpl(parent, null, owner, mode, 0, lastModifiedDate);
			Assert.fail("Expected an exception, but none was thrown.");
			return;
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		}
		try {
			file = new FileMetaDataImpl(parent, "/", owner, mode, 0, lastModifiedDate);
			Assert.fail("Expected an exception, but none was thrown.");
			return;
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		}
		try {
			file = new FileMetaDataImpl(parent, "usr", null, mode, 0, lastModifiedDate);
			Assert.fail("Expected an exception, but none was thrown.");
			return;
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		}
		try {
			file = new FileMetaDataImpl(parent, "usr", owner, null, 0, lastModifiedDate);
			Assert.fail("Expected an exception, but none was thrown.");
			return;
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		}
		try {
			file = new FileMetaDataImpl(parent, "usr", owner, null, -2, lastModifiedDate);
			Assert.fail("Expected an exception, but none was thrown.");
			return;
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		}
		try {
			file = new FileMetaDataImpl(parent, "usr", owner, mode, 0, null);
			Assert.fail("Expected an exception, but none was thrown.");
			return;
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		}


		// --- Executable file ---
		file = new FileMetaDataImpl(parent, "myprogram", owner, mode, 12345, lastModifiedDate);
		Assert.assertEquals("/usr/local/mypackage/", file.getPath());
		Assert.assertEquals("myprogram", file.getName());
		Assert.assertEquals("/usr/local/mypackage/myprogram", file.getAbsolutePath());
		Assert.assertTrue(file.isFile());
		Assert.assertFalse(file.isDirectory());
		Assert.assertFalse(file.isSymbolicLink());
		Assert.assertNull(file.getTargetPath());
		Assert.assertSame(owner, file.getOwner());
		Assert.assertSame(mode, file.getMode());
		Assert.assertEquals("-rwxr-xr-x", file.getModeAsText());
		Assert.assertEquals(12345, file.getLength());
		Assert.assertSame(lastModifiedDate, file.getLastModifiedDate());


		// --- Normal file ---
		mode = new FileModeImpl(0644);
		file = new FileMetaDataImpl(parent, "myprogram", owner, mode, 12345, lastModifiedDate);
		Assert.assertEquals("/usr/local/mypackage/", file.getPath());
		Assert.assertEquals("myprogram", file.getName());
		Assert.assertEquals("/usr/local/mypackage/myprogram", file.getAbsolutePath());
		Assert.assertTrue(file.isFile());
		Assert.assertFalse(file.isDirectory());
		Assert.assertFalse(file.isSymbolicLink());
		Assert.assertNull(file.getTargetPath());
		Assert.assertSame(owner, file.getOwner());
		Assert.assertSame(mode, file.getMode());
		Assert.assertEquals("-rw-r--r--", file.getModeAsText());
		Assert.assertEquals(12345, file.getLength());
		Assert.assertSame(lastModifiedDate, file.getLastModifiedDate());
	}


	/**
	 * <p>
	 * Performs some tests on the
	 * {@link FileMetaDataImpl#FileMetaDataImpl(String, String, String, FileOwner, FileMode, Date)}
	 * method.
	 * </p>
	 */
	@Test
	public void testFileMetaDataImplStringStringStringFileOwnerFileModeDate() {
		FileMetaData file;
		FileOwner owner;
		FileMode mode;
		Date lastModifiedDate;
		
		
		// --- Parameters ---
		owner = new FileOwnerImpl(0, "root", 0, "root");
		mode = new FileModeImpl(00755);
		lastModifiedDate = new Date(1463043896603L);
		try {
			file = new FileMetaDataImpl((String) null, "usr", "/mnt/usr", owner, mode, lastModifiedDate);
			Assert.fail("Expected an exception, but none was thrown.");
			return;
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		}
		try {
			file = new FileMetaDataImpl("", "usr", "/mnt/usr", owner, mode, lastModifiedDate);
			Assert.fail("Expected an exception, but none was thrown.");
			return;
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		}
		try {
			file = new FileMetaDataImpl("/", null, "/mnt/usr", owner, mode, lastModifiedDate);
			Assert.fail("Expected an exception, but none was thrown.");
			return;
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		}
		try {
			file = new FileMetaDataImpl("/", "/", "/mnt/usr", owner, mode, lastModifiedDate);
			Assert.fail("Expected an exception, but none was thrown.");
			return;
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		}
		try {
			file = new FileMetaDataImpl("/", "/", null, owner, mode, lastModifiedDate);
			Assert.fail("Expected an exception, but none was thrown.");
			return;
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		}
		try {
			file = new FileMetaDataImpl("/", "/", "", owner, mode, lastModifiedDate);
			Assert.fail("Expected an exception, but none was thrown.");
			return;
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		}
		try {
			file = new FileMetaDataImpl("/", "", "/mnt/usr", null, mode, lastModifiedDate);
			Assert.fail("Expected an exception, but none was thrown.");
			return;
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		}
		try {
			file = new FileMetaDataImpl("/", "usr", "/mnt/usr", owner, null, lastModifiedDate);
			Assert.fail("Expected an exception, but none was thrown.");
			return;
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		}
		try {
			file = new FileMetaDataImpl("/", "usr", "/mnt/usr", owner, mode, null);
			Assert.fail("Expected an exception, but none was thrown.");
			return;
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		}
		
		
		// --- Symbolic link /usr ---
		file = new FileMetaDataImpl("./", "usr", "/mnt/usr", owner, mode, lastModifiedDate);
		Assert.assertEquals("/", file.getPath());
		Assert.assertEquals("usr", file.getName());
		Assert.assertEquals("/usr", file.getAbsolutePath());
		Assert.assertTrue(file.isFile());
		Assert.assertFalse(file.isDirectory());
		Assert.assertTrue(file.isSymbolicLink());
		Assert.assertEquals("/mnt/usr", file.getTargetPath());
		Assert.assertSame(owner, file.getOwner());
		Assert.assertSame(mode, file.getMode());
		Assert.assertEquals("lrwxr-xr-x", file.getModeAsText());
		Assert.assertEquals(-1, file.getLength());
		Assert.assertSame(lastModifiedDate, file.getLastModifiedDate());
		
		
		// --- Symbolic link /usr/local ---
		file = new FileMetaDataImpl("usr", "local", "/mnt/usr/local", owner, mode, lastModifiedDate);
		Assert.assertEquals("/usr/", file.getPath());
		Assert.assertEquals("local", file.getName());
		Assert.assertEquals("/usr/local", file.getAbsolutePath());
		Assert.assertTrue(file.isFile());
		Assert.assertFalse(file.isDirectory());
		Assert.assertTrue(file.isSymbolicLink());
		Assert.assertEquals("/mnt/usr/local", file.getTargetPath());
		Assert.assertSame(owner, file.getOwner());
		Assert.assertSame(mode, file.getMode());
		Assert.assertEquals("lrwxr-xr-x", file.getModeAsText());
		Assert.assertEquals(-1, file.getLength());
		Assert.assertSame(lastModifiedDate, file.getLastModifiedDate());
	}
	
	
	/**
	 * <p>
	 * Performs some tests on the
	 * {@link FileMetaDataImpl#FileMetaDataImpl(FileMetaData, String, String, FileOwner, FileMode, Date)}
	 * method.
	 * </p>
	 */
	@Test
	public void testFileMetaDataImplFileMetaDataStringStringFileOwnerFileModeDate2() {
		FileMetaData file, parent;
		FileOwner owner;
		FileMode mode;
		Date lastModifiedDate;
		
		
		// --- Parameters ---
		owner = new FileOwnerImpl(0, "root", 0, "root");
		mode = new FileModeImpl(00755);
		lastModifiedDate = new Date(1463043896603L);
		parent = new FileMetaDataImpl("/", "", new FileOwnerImpl(0, "root", 0, "root"), new FileModeImpl(00755),
				new Date(1463043896603L));
		try {
			file = new FileMetaDataImpl((FileMetaData) null, "usr", "/mnt/usr", owner, mode, lastModifiedDate);
			Assert.fail("Expected an exception, but none was thrown.");
			return;
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		}
		try {
			file = new FileMetaDataImpl(parent, null, "/mnt/usr", owner, mode, lastModifiedDate);
			Assert.fail("Expected an exception, but none was thrown.");
			return;
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		}
		try {
			file = new FileMetaDataImpl(parent, "/", "/mnt/usr", owner, mode, lastModifiedDate);
			Assert.fail("Expected an exception, but none was thrown.");
			return;
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		}
		try {
			file = new FileMetaDataImpl(parent, "usr", null, owner, mode, lastModifiedDate);
			Assert.fail("Expected an exception, but none was thrown.");
			return;
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		}
		try {
			file = new FileMetaDataImpl(parent, "usr", "", owner, mode, lastModifiedDate);
			Assert.fail("Expected an exception, but none was thrown.");
			return;
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		}
		try {
			file = new FileMetaDataImpl(parent, "usr", "/mnt/usr", null, mode, lastModifiedDate);
			Assert.fail("Expected an exception, but none was thrown.");
			return;
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		}
		try {
			file = new FileMetaDataImpl(parent, "usr", "/mnt/usr", owner, null, lastModifiedDate);
			Assert.fail("Expected an exception, but none was thrown.");
			return;
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		}
		try {
			file = new FileMetaDataImpl(parent, "usr", "/mnt/usr", owner, mode, null);
			Assert.fail("Expected an exception, but none was thrown.");
			return;
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		}


		// --- Symbolic link /usr ---
		file = new FileMetaDataImpl(parent, "usr", "/mnt/usr", owner, mode, lastModifiedDate);
		Assert.assertEquals("/", file.getPath());
		Assert.assertEquals("usr", file.getName());
		Assert.assertEquals("/usr", file.getAbsolutePath());
		Assert.assertTrue(file.isFile());
		Assert.assertFalse(file.isDirectory());
		Assert.assertTrue(file.isSymbolicLink());
		Assert.assertEquals("/mnt/usr", file.getTargetPath());
		Assert.assertSame(owner, file.getOwner());
		Assert.assertSame(mode, file.getMode());
		Assert.assertEquals("lrwxr-xr-x", file.getModeAsText());
		Assert.assertEquals(-1, file.getLength());
		Assert.assertSame(lastModifiedDate, file.getLastModifiedDate());
		
		
		// --- Symbolic link /usr/local ---
		parent = new FileMetaDataImpl(parent, "usr", owner, mode, lastModifiedDate);
		file = new FileMetaDataImpl(parent, "local", "/mnt/usr/local", owner, mode, lastModifiedDate);
		Assert.assertEquals("/usr/", file.getPath());
		Assert.assertEquals("local", file.getName());
		Assert.assertEquals("/usr/local", file.getAbsolutePath());
		Assert.assertTrue(file.isFile());
		Assert.assertFalse(file.isDirectory());
		Assert.assertTrue(file.isSymbolicLink());
		Assert.assertEquals("/mnt/usr/local", file.getTargetPath());
		Assert.assertSame(owner, file.getOwner());
		Assert.assertSame(mode, file.getMode());
		Assert.assertEquals("lrwxr-xr-x", file.getModeAsText());
		Assert.assertEquals(-1, file.getLength());
		Assert.assertSame(lastModifiedDate, file.getLastModifiedDate());
	}
	
	
	/**
	 * <p>
	 * Performs some tests on the
	 * {@link FileMetaDataImpl#createDirectoryMetaData(String, FileOwner, FileMode, Date)}
	 * method.
	 * </p>
	 */
	@Test
	public void testCreateDirectoryMetaData() {
		FileMetaData file;
		FileOwner owner;
		FileMode mode;
		Date lastModifiedDate;


		// --- Parameters ---
		owner = new FileOwnerImpl(0, "root", 0, "root");
		mode = new FileModeImpl(0755);
		lastModifiedDate = new Date(1463043896603L);
		try {
			FileMetaDataImpl.createDirectoryMetaData(null, owner, mode, lastModifiedDate);
			Assert.fail("Expected an exception, but none was thrown.");
			return;
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		}
		try {
			FileMetaDataImpl.createDirectoryMetaData("", owner, mode, lastModifiedDate);
			Assert.fail("Expected an exception, but none was thrown.");
			return;
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		}
		try {
			FileMetaDataImpl.createDirectoryMetaData("/", null, mode, lastModifiedDate);
			Assert.fail("Expected an exception, but none was thrown.");
			return;
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		}
		try {
			FileMetaDataImpl.createDirectoryMetaData("/", owner, null, lastModifiedDate);
			Assert.fail("Expected an exception, but none was thrown.");
			return;
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		}
		try {
			FileMetaDataImpl.createDirectoryMetaData("/", owner, mode, null);
			Assert.fail("Expected an exception, but none was thrown.");
			return;
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		}
		
		
		// --- Root ---
		file = FileMetaDataImpl.createDirectoryMetaData("/", owner, mode, lastModifiedDate);
		Assert.assertEquals("/", file.getPath());
		Assert.assertEquals("", file.getName());
		Assert.assertEquals("/", file.getAbsolutePath());
		Assert.assertFalse(file.isFile());
		Assert.assertTrue(file.isDirectory());
		Assert.assertFalse(file.isSymbolicLink());
		Assert.assertNull(file.getTargetPath());
		Assert.assertSame(owner, file.getOwner());
		Assert.assertSame(mode, file.getMode());
		Assert.assertEquals("drwxr-xr-x", file.getModeAsText());
		Assert.assertEquals(-1, file.getLength());
		Assert.assertSame(lastModifiedDate, file.getLastModifiedDate());
		
		
		// --- ./usr/ ---
		file = FileMetaDataImpl.createDirectoryMetaData("./usr/", owner, mode, lastModifiedDate);
		Assert.assertEquals("/", file.getPath());
		Assert.assertEquals("usr", file.getName());
		Assert.assertEquals("/usr/", file.getAbsolutePath());
		Assert.assertFalse(file.isFile());
		Assert.assertTrue(file.isDirectory());
		Assert.assertFalse(file.isSymbolicLink());
		Assert.assertNull(file.getTargetPath());
		Assert.assertSame(owner, file.getOwner());
		Assert.assertSame(mode, file.getMode());
		Assert.assertEquals("drwxr-xr-x", file.getModeAsText());
		Assert.assertEquals(-1, file.getLength());
		Assert.assertSame(lastModifiedDate, file.getLastModifiedDate());
		
		
		// --- /usr/ ---
		file = FileMetaDataImpl.createDirectoryMetaData("/usr/", owner, mode, lastModifiedDate);
		Assert.assertEquals("/", file.getPath());
		Assert.assertEquals("usr", file.getName());
		Assert.assertEquals("/usr/", file.getAbsolutePath());
		Assert.assertFalse(file.isFile());
		Assert.assertTrue(file.isDirectory());
		Assert.assertFalse(file.isSymbolicLink());
		Assert.assertNull(file.getTargetPath());
		Assert.assertSame(owner, file.getOwner());
		Assert.assertSame(mode, file.getMode());
		Assert.assertEquals("drwxr-xr-x", file.getModeAsText());
		Assert.assertEquals(-1, file.getLength());
		Assert.assertSame(lastModifiedDate, file.getLastModifiedDate());
		
		
		// --- usr/ ---
		file = FileMetaDataImpl.createDirectoryMetaData("usr/", owner, mode, lastModifiedDate);
		Assert.assertEquals("/", file.getPath());
		Assert.assertEquals("usr", file.getName());
		Assert.assertEquals("/usr/", file.getAbsolutePath());
		Assert.assertFalse(file.isFile());
		Assert.assertTrue(file.isDirectory());
		Assert.assertFalse(file.isSymbolicLink());
		Assert.assertNull(file.getTargetPath());
		Assert.assertSame(owner, file.getOwner());
		Assert.assertSame(mode, file.getMode());
		Assert.assertEquals("drwxr-xr-x", file.getModeAsText());
		Assert.assertEquals(-1, file.getLength());
		Assert.assertSame(lastModifiedDate, file.getLastModifiedDate());
		
		
		// --- /usr ---
		file = FileMetaDataImpl.createDirectoryMetaData("/usr", owner, mode, lastModifiedDate);
		Assert.assertEquals("/", file.getPath());
		Assert.assertEquals("usr", file.getName());
		Assert.assertEquals("/usr/", file.getAbsolutePath());
		Assert.assertFalse(file.isFile());
		Assert.assertTrue(file.isDirectory());
		Assert.assertFalse(file.isSymbolicLink());
		Assert.assertNull(file.getTargetPath());
		Assert.assertSame(owner, file.getOwner());
		Assert.assertSame(mode, file.getMode());
		Assert.assertEquals("drwxr-xr-x", file.getModeAsText());
		Assert.assertEquals(-1, file.getLength());
		Assert.assertSame(lastModifiedDate, file.getLastModifiedDate());
		
		
		// --- /usr/local/ ---
		file = FileMetaDataImpl.createDirectoryMetaData("/usr/local/", owner, mode, lastModifiedDate);
		Assert.assertEquals("/usr/", file.getPath());
		Assert.assertEquals("local", file.getName());
		Assert.assertEquals("/usr/local/", file.getAbsolutePath());
		Assert.assertFalse(file.isFile());
		Assert.assertTrue(file.isDirectory());
		Assert.assertFalse(file.isSymbolicLink());
		Assert.assertNull(file.getTargetPath());
		Assert.assertSame(owner, file.getOwner());
		Assert.assertSame(mode, file.getMode());
		Assert.assertEquals("drwxr-xr-x", file.getModeAsText());
		Assert.assertEquals(-1, file.getLength());
		Assert.assertSame(lastModifiedDate, file.getLastModifiedDate());
	}
	
	
	/**
	 * <p>
	 * Performs some tests on the
	 * {@link FileMetaDataImpl#createFileMetaData(String, FileOwner, FileMode, long, Date)}
	 * method.
	 * </p>
	 */
	@Test
	public void testCreateFileMetaData() {
		FileMetaData file;
		FileOwner owner;
		FileMode mode;
		Date lastModifiedDate;


		// --- Parameters ---
		owner = new FileOwnerImpl(0, "root", 0, "root");
		mode = new FileModeImpl(0755);
		lastModifiedDate = new Date(1463043896603L);
		try {
			file = FileMetaDataImpl.createFileMetaData(null, owner, mode, 0, lastModifiedDate);
			Assert.fail("Expected an exception, but none was thrown.");
			return;
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		}
		try {
			file = FileMetaDataImpl.createFileMetaData("", owner, mode, 0, lastModifiedDate);
			Assert.fail("Expected an exception, but none was thrown.");
			return;
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		}
		try {
			file = FileMetaDataImpl.createFileMetaData("/usr/", owner, mode, 0, lastModifiedDate);
			Assert.fail("Expected an exception, but none was thrown.");
			return;
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		}
		try {
			file = FileMetaDataImpl.createFileMetaData("/usr/", owner, mode, 0, lastModifiedDate);
			Assert.fail("Expected an exception, but none was thrown.");
			return;
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		}
		try {
			file = FileMetaDataImpl.createFileMetaData("/usr/local/mypackage/myprogram", null, mode, 0, lastModifiedDate);
			Assert.fail("Expected an exception, but none was thrown.");
			return;
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		}
		try {
			file = FileMetaDataImpl.createFileMetaData("/usr/local/mypackage/myprogram", owner, null, -2, lastModifiedDate);
			Assert.fail("Expected an exception, but none was thrown.");
			return;
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		}
		try {
			file = FileMetaDataImpl.createFileMetaData("/usr/local/mypackage/myprogram", owner, mode, 0, null);
			Assert.fail("Expected an exception, but none was thrown.");
			return;
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		}
		
		
		// --- File ---
		file = FileMetaDataImpl.createFileMetaData("/usr/local/mypackage/myprogram", owner, mode, 12345, lastModifiedDate);
		Assert.assertEquals("/usr/local/mypackage/", file.getPath());
		Assert.assertEquals("myprogram", file.getName());
		Assert.assertEquals("/usr/local/mypackage/myprogram", file.getAbsolutePath());
		Assert.assertTrue(file.isFile());
		Assert.assertFalse(file.isDirectory());
		Assert.assertFalse(file.isSymbolicLink());
		Assert.assertNull(file.getTargetPath());
		Assert.assertSame(owner, file.getOwner());
		Assert.assertSame(mode, file.getMode());
		Assert.assertEquals("-rwxr-xr-x", file.getModeAsText());
		Assert.assertEquals(12345, file.getLength());
		Assert.assertSame(lastModifiedDate, file.getLastModifiedDate());
	}
	
	
	/**
	 * <p>
	 * Performs some tests on the
	 * {@link FileMetaDataImpl#createSymbolicLinkMetaData(String, String, FileOwner, FileMode, Date)}
	 * method.
	 * </p>
	 */
	@Test
	public void testCreateSymbolicLinkMetaData() {
		FileMetaData file;
		FileOwner owner;
		FileMode mode;
		Date lastModifiedDate;
		
		
		// --- Parameters ---
		owner = new FileOwnerImpl(0, "root", 0, "root");
		mode = new FileModeImpl(0755);
		lastModifiedDate = new Date(1463043896603L);
		try {
			file = FileMetaDataImpl.createSymbolicLinkMetaData(null, "/mnt/usr/local", owner, mode, lastModifiedDate);
			Assert.fail("Expected an exception, but none was thrown.");
			return;
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		}
		try {
			file = FileMetaDataImpl.createSymbolicLinkMetaData("", "/mnt/usr/local", owner, mode, lastModifiedDate);
			Assert.fail("Expected an exception, but none was thrown.");
			return;
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		}
		try {
			file = FileMetaDataImpl.createSymbolicLinkMetaData("/usr/local", null, owner, mode, lastModifiedDate);
			Assert.fail("Expected an exception, but none was thrown.");
			return;
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		}
		try {
			file = FileMetaDataImpl.createSymbolicLinkMetaData("/usr/local", "", owner, mode, lastModifiedDate);
			Assert.fail("Expected an exception, but none was thrown.");
			return;
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		}
		try {
			file = FileMetaDataImpl.createSymbolicLinkMetaData("/usr/local", "/mnt/usr/local", null, mode, lastModifiedDate);
			Assert.fail("Expected an exception, but none was thrown.");
			return;
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		}
		try {
			file = FileMetaDataImpl.createSymbolicLinkMetaData("/usr/local", "/mnt/usr/local", owner, null, lastModifiedDate);
			Assert.fail("Expected an exception, but none was thrown.");
			return;
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		}
		try {
			file = FileMetaDataImpl.createSymbolicLinkMetaData("/usr/local", "/mnt/usr/local", owner, mode, null);
			Assert.fail("Expected an exception, but none was thrown.");
			return;
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		}
		
		
		// --- Symbolic link ---
		file = FileMetaDataImpl.createSymbolicLinkMetaData("/usr/local", "/mnt/usr/local", owner, mode, lastModifiedDate);
		Assert.assertEquals("/usr/", file.getPath());
		Assert.assertEquals("local", file.getName());
		Assert.assertEquals("/usr/local", file.getAbsolutePath());
		Assert.assertTrue(file.isFile());
		Assert.assertFalse(file.isDirectory());
		Assert.assertTrue(file.isSymbolicLink());
		Assert.assertEquals("/mnt/usr/local", file.getTargetPath());
		Assert.assertSame(owner, file.getOwner());
		Assert.assertSame(mode, file.getMode());
		Assert.assertEquals("lrwxr-xr-x", file.getModeAsText());
		Assert.assertEquals(-1, file.getLength());
		Assert.assertSame(lastModifiedDate, file.getLastModifiedDate());
	}


}
