/*
 * dpkg - Debian Package library and the Debian Package Maven plugin
 * (c) Copyright 2015 Gerrit Hohl
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package net.sourceforge.javadpkg.impl;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import org.junit.Assert;
import org.junit.Ignore;
import org.junit.Test;

import net.sourceforge.javadpkg.AbstractDpkgTest;
import net.sourceforge.javadpkg.Context;
import net.sourceforge.javadpkg.DebianPackage;
import net.sourceforge.javadpkg.ParseException;
import net.sourceforge.javadpkg.Warning;
import net.sourceforge.javadpkg.control.Control;
import net.sourceforge.javadpkg.control.impl.BinaryControlImpl;
import net.sourceforge.javadpkg.io.DataSource;
import net.sourceforge.javadpkg.io.impl.DataFileSource;


/**
 * <p>
 * Performs some tests on the {@link DebianPackageParserImpl} class.
 * </p>
 *
 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
 * @version <b>1.0</b>, 27.12.2015 by Gerrit Hohl
 */
public class DebianPackageParserImplTest extends AbstractDpkgTest {
	
	
	/**
	 * <p>
	 * Performs some tests on the parser by parsing the Debian packets of some
	 * Ubuntu distributions.
	 * </p>
	 */
	@Test
	@Ignore
	public void test() {
		File folder;
		List<File> files;
		DebianPackageParserImpl parser;
		List<Warning> allWarnings, warnings;
		Context context;
		DebianPackageParseCollectHandler handler;
		DebianPackage deb;
		Control control;


		folder = new File(System.getProperty("user.home"), "Desktop/Ubuntu");
		Assert.assertTrue("Directory |" + folder.getAbsolutePath() + "| doesn't exist.",
				folder.exists() && folder.isDirectory());
		files = this.getDebFiles(folder);
		System.out.println(files.size() + " files found.");

		parser = new DebianPackageParserImpl();
		allWarnings = new ArrayList<>();
		for (File file : files) {
			System.out.println(file.getAbsolutePath());
			context = new ContextImpl();
			handler = new DebianPackageParseCollectHandler();
			try {
				try (DataSource source = new DataFileSource(file)) {
					deb = parser.parseDebianPackage(source, handler, context);
				}
			} catch (IOException | ParseException e) {
				e.printStackTrace();
				Assert.fail("Error while reading file |" + file.getAbsolutePath() + "|: " + e.getMessage());
				return;
			}
			Assert.assertNotNull(deb);
			Assert.assertEquals("2.0", deb.getFileFormatVersion());

			control = deb.getControl();
			Assert.assertNotNull(control);
			Assert.assertSame(BinaryControlImpl.class, control.getClass());

			handler.printFiles(System.out);
			
			warnings = context.getWarnings();
			if (!warnings.isEmpty()) {
				allWarnings.addAll(warnings);
			}
		}
		this.printWarnings(allWarnings);
		Assert.fail("Not yet implemented");
	}


	/**
	 * <p>
	 * Returns the Debian packet files of the specified folder and its
	 * sub-folders.
	 * </p>
	 *
	 * @param folder
	 *            The folder.
	 * @return The Debian packet files.
	 */
	private List<File> getDebFiles(File folder) {
		List<File> fileList;
		File[] files;


		System.out.println(folder.getAbsolutePath());
		files = folder.listFiles();
		Assert.assertNotNull(files);
		fileList = new ArrayList<>();
		for (File file : files) {
			if (file.isFile() && file.getName().endsWith(".deb")) {
				fileList.add(file);
			} else if (file.isDirectory()) {
				fileList.addAll(this.getDebFiles(file));
			}
		}
		return fileList;
	}


	/**
	 * <p>
	 * Prints the warnings to the standard output.
	 * </p>
	 *
	 * @param warnings
	 *            The warnings.
	 */
	private void printWarnings(List<Warning> warnings) {
		Map<Class<?>, List<Warning>> warnMap;
		List<Warning> warns;
		List<Class<?>> classes;
		
		
		warnMap = new LinkedHashMap<>();
		for (Warning warning : warnings) {
			warns = warnMap.get(warning.getClass());
			if (warns == null) {
				warns = new ArrayList<>();
				warnMap.put(warning.getClass(), warns);
			}
			warns.add(warning);
		}
		classes = new ArrayList<>(warnMap.keySet());
		Collections.sort(classes, new ClassComparator());
		for (Class<?> clazz : classes) {
			System.out.println(clazz.getSimpleName());
			warns = warnMap.get(clazz);
			for (Warning warning : warns) {
				System.out.println("\t" + warning.getText());
			}
		}
	}
	
	
	/* **********************************************************************
	 * **********************************************************************
	 * **********************************************************************
	 * **********************************************************************
	 * **********************************************************************
	 */
	
	
	/**
	 * <p>
	 * A comparator for classes.
	 * </p>
	 *
	 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
	 * @version <b>1.0</b>, 12.05.2016 by Gerrit Hohl
	 */
	private class ClassComparator implements Comparator<Class<?>> {
		
		
		/**
		 * <p>
		 * Creates the comparator.
		 * </p>
		 */
		public ClassComparator() {
			super();
		}
		
		
		@Override
		public int compare(Class<?> o1, Class<?> o2) {
			return o1.getSimpleName().compareTo(o2.getSimpleName());
		}
		
		
	}
	
	
}
