/*
 * dpkg - Debian Package library and the Debian Package Maven plugin
 * (c) Copyright 2016 Gerrit Hohl
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package net.sourceforge.javadpkg.impl;

import java.io.IOException;
import java.io.PrintStream;
import java.util.ArrayList;
import java.util.List;

import net.sourceforge.javadpkg.DebianPackageParseHandler;
import net.sourceforge.javadpkg.ParseException;
import net.sourceforge.javadpkg.io.DataSource;
import net.sourceforge.javadpkg.io.FileMetaData;

/**
 * <p>
 * A {@link DebianPackageParseHandler} which simply collects all information.
 * </p>
 *
 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
 * @version <b>1.0</b>, 12.05.2016 by Gerrit Hohl
 */
public class DebianPackageParseCollectHandler implements DebianPackageParseHandler {
	
	
	/** The files. */
	private List<FileMetaData> files;


	/**
	 * <p>
	 * Creates a handler.
	 * </p>
	 */
	public DebianPackageParseCollectHandler() {
		super();

		this.files = new ArrayList<>();
	}


	@Override
	public void handleData(FileMetaData metaData, DataSource source) throws IOException, ParseException {
		if (metaData == null)
			throw new IllegalArgumentException("Argument metaData is null.");
		if (metaData.isFile() && (source == null))
			throw new IllegalArgumentException("Argument metaData is a file, but argument source is null.");
		
		this.files.add(metaData);
	}
	
	
	/**
	 * <p>
	 * Returns the files which have been passed to the handler by the
	 * {@link #handleData(FileMetaData, DataSource)} method.
	 * </p>
	 *
	 * @return The files.
	 */
	public List<FileMetaData> getFiles() {
		return (new ArrayList<>(this.files));
	}
	
	
	/**
	 * <p>
	 * Prints the files into the specified stream.
	 * </p>
	 *
	 * @param out
	 *            The stream.
	 * @throws IllegalArgumentException
	 *             If the stream is <code>null</code>.
	 */
	public void printFiles(PrintStream out) {
		Long groupId, userId, length;


		if (out == null)
			throw new IllegalArgumentException("Argument out is null.");
		
		for (FileMetaData file : this.files) {
			groupId = Long.valueOf(file.getOwner().getGroupId());
			userId = Long.valueOf(file.getOwner().getUserId());
			if (file.isDirectory() && (file.getLength() < 0)) {
				length = Long.valueOf(0);
			} else {
				length = Long.valueOf(file.getLength());
			}

			out.println(String.format("%s %5d %-8s %5d %-8s %10d %s", file.getModeAsText(), groupId,
					file.getOwner().getGroupName(), userId, file.getOwner().getUserName(), length, file.getAbsolutePath()));
		}
	}
	
	
}
