/*
 * dpkg - Debian Package library and the Debian Package Maven plugin
 * (c) Copyright 2016 Gerrit Hohl
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package net.sourceforge.javadpkg.impl;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import net.sourceforge.javadpkg.BuildException;
import net.sourceforge.javadpkg.ChangeLog;
import net.sourceforge.javadpkg.Context;
import net.sourceforge.javadpkg.Copyright;
import net.sourceforge.javadpkg.DebianPackageBuilder;
import net.sourceforge.javadpkg.Script;
import net.sourceforge.javadpkg.control.Control;
import net.sourceforge.javadpkg.control.Size;
import net.sourceforge.javadpkg.io.DataSource;
import net.sourceforge.javadpkg.io.DataTarget;
import net.sourceforge.javadpkg.io.FileMode;
import net.sourceforge.javadpkg.io.FileOwner;

/**
 * <p>
 * A mock implementation of the {@link DebianPackageBuilder}.
 * </p>
 *
 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
 * @version <b>1.0</b>, 20.05.2016 by Gerrit Hohl
 */
public class DebianPackageBuilderMock implements DebianPackageBuilder {
	
	
	/** The control. */
	private Control					control;
	/** The overhead if the installed size. */
	private Size					installedSizeOverhead;
	/** The pre-installation script. */
	private Script					preInstall;
	/** The post-installation script. */
	private Script					postInstall;
	/** The pre-removal script. */
	private Script					preRemove;
	/** The post-removal script. */
	private Script					postRemove;
	/** The added data. */
	private Map<String, DataNode>	dataMap;
	/** The copyright. */
	private Copyright				copyright;
	/** The change log. */
	private ChangeLog				changeLog;
	
	
	/**
	 * <p>
	 * Creates a builder.
	 * </p>
	 */
	public DebianPackageBuilderMock() {
		super();
		
		this.control = null;
		this.installedSizeOverhead = null;
		this.preInstall = null;
		this.postInstall = null;
		this.preRemove = null;
		this.postRemove = null;
		this.dataMap = new HashMap<>();
		this.copyright = null;
		this.changeLog = null;
	}
	
	
	@Override
	public void setControl(Control control) {
		this.control = control;
	}
	
	
	/**
	 * <p>
	 * Returns the control.
	 * </p>
	 *
	 * @return The control or <code>null</code>, if no control is set.
	 */
	public Control getControl() {
		return this.control;
	}
	
	
	@Override
	public void setInstalledSizeOverhead(Size installedSizeOverhead) {
		this.installedSizeOverhead = installedSizeOverhead;
	}
	
	
	/**
	 * <p>
	 * Returns the overhead if the installed size.
	 * </p>
	 *
	 * @return The overhead or <code>null</code>, if no overhead is set.
	 */
	public Size getInstalledSizeOverhead() {
		return this.installedSizeOverhead;
	}
	
	
	@Override
	public void setPreInstall(Script preInstall) {
		this.preInstall = preInstall;
	}
	
	
	/**
	 * <p>
	 * Returns the pre-installation script.
	 * </p>
	 *
	 * @return The script or <code>null</code>, if no script is set.
	 */
	public Script getPreInstall() {
		return this.preInstall;
	}
	
	
	@Override
	public void setPostInstall(Script postInstall) {
		this.postInstall = postInstall;
	}
	
	
	/**
	 * <p>
	 * Returns the post-installation script.
	 * </p>
	 *
	 * @return The script or <code>null</code>, if no script is set.
	 */
	public Script getPostInstall() {
		return this.postInstall;
	}
	
	
	@Override
	public void setPreRemove(Script preRemove) {
		this.preRemove = preRemove;
	}
	
	
	/**
	 * <p>
	 * Returns the pre-removal script.
	 * </p>
	 *
	 * @return The script or <code>null</code>, if no script is set.
	 */
	public Script getPreRemove() {
		return this.preRemove;
	}
	
	
	@Override
	public void setPostRemove(Script postRemove) {
		this.postRemove = postRemove;
	}
	
	
	/**
	 * <p>
	 * Returns the post-removal script.
	 * </p>
	 *
	 * @return The script or <code>null</code>, if no script is set.
	 */
	public Script getPostRemove() {
		return this.postRemove;
	}
	
	
	@Override
	public void addDataDirectory(String path) {
		DataNode data;


		if (path == null)
			throw new IllegalArgumentException("Argument path is null.");
		
		data = new DataNode(path, null, null);
		if (this.dataMap.put(path, data) != null)
			throw new IllegalArgumentException("Data with path |" + path + "| already exists.");
	}
	
	
	@Override
	public void addDataDirectory(String path, FileOwner owner, FileMode mode) {
		DataNode data;


		if (path == null)
			throw new IllegalArgumentException("Argument path is null.");
		if (owner == null)
			throw new IllegalArgumentException("Argument owner is null.");
		if (mode == null)
			throw new IllegalArgumentException("Argument mode is null.");
		
		data = new DataNode(path, owner, mode);
		if (this.dataMap.put(path, data) != null)
			throw new IllegalArgumentException("Data with path |" + path + "| already exists.");
	}
	
	
	@Override
	public void addDataFile(DataSource source, String path) {
		DataNode data;


		if (source == null)
			throw new IllegalArgumentException("Argument source is null.");
		if (path == null)
			throw new IllegalArgumentException("Argument path is null.");
		
		data = new DataNode(source, path, null, null);
		if (this.dataMap.put(path, data) != null)
			throw new IllegalArgumentException("Data with path |" + path + "| already exists.");
	}
	
	
	@Override
	public void addDataFile(DataSource source, String path, FileOwner owner, FileMode mode) {
		DataNode data;


		if (source == null)
			throw new IllegalArgumentException("Argument source is null.");
		if (path == null)
			throw new IllegalArgumentException("Argument path is null.");
		if (owner == null)
			throw new IllegalArgumentException("Argument owner is null.");
		if (mode == null)
			throw new IllegalArgumentException("Argument mode is null.");
		
		data = new DataNode(source, path, owner, mode);
		if (this.dataMap.put(path, data) != null)
			throw new IllegalArgumentException("Data with path |" + path + "| already exists.");
	}


	@Override
	public void addDataSymLink(String path, String target, FileOwner owner, FileMode mode) {
		DataNode data;
		
		
		if (path == null)
			throw new IllegalArgumentException("Argument path is null.");
		if (target == null)
			throw new IllegalArgumentException("Argument target is null.");
		if (owner == null)
			throw new IllegalArgumentException("Argument owner is null.");
		if (mode == null)
			throw new IllegalArgumentException("Argument mode is null.");
		
		data = new DataNode(path, target, owner, mode);
		if (this.dataMap.put(path, data) != null)
			throw new IllegalArgumentException("Data with path |" + path + "| already exists.");
	}
	
	
	/**
	 * <p>
	 * Returns the data.
	 * </p>
	 * <p>
	 * The data is sorted by the paths. Directories have a higher priority than
	 * files.
	 * </p>
	 *
	 * @return The data.
	 */
	public List<DataNode> getDataNodes() {
		List<DataNode> result;
		
		
		result = new ArrayList<>(this.dataMap.values());
		Collections.sort(result);
		return result;
	}
	
	
	/**
	 * <p>
	 * Returns a string representation of the data.
	 * </p>
	 *
	 * @return The string representation.
	 */
	public String getDataNodesAsString() {
		StringBuilder sb;
		List<DataNode> nodes;
		FileOwner owner;
		Long groupId, userId, size;
		String source, target;
		
		
		sb = new StringBuilder();
		nodes = this.getDataNodes();
		for (DataNode node : nodes) {
			if (sb.length() > 0) {
				sb.append('\n');
			}
			
			owner = node.getOwner();
			
			groupId = Long.valueOf(owner.getGroupId());
			userId = Long.valueOf(owner.getUserId());
			size = null;
			source = null;
			target = null;
			
			// --- Directory or regular file? ---
			if (node.isDirectory()) {
				sb.append('d');
			} else {
				if (!node.isSymLink()) {
					size = Long.valueOf(node.getSource().getLength());
					source = node.getSource().getName();
				}
				if (node.isSymLink()) {
					target = node.getTarget();
					sb.append('l');
				} else {
					sb.append('-');
				}
			}
			// --- Rights of owner, group and other ---
			sb.append(node.getMode().getText());
			
			// --- Format the rest of the information ---
			sb.append(String.format(" %5d %-8s %5d %-8s %10d %-40s", groupId, owner.getGroupName(), userId, owner.getUserName(),
					size, node.getPath()));
			if (source != null) {
				sb.append(String.format(" (Source: |%s|)", source));
			} else if (target != null) {
				sb.append(String.format(" (Target: |%s|)", target));
			}
		}
		return sb.toString();
	}
	
	
	@Override
	public void setCopyright(Copyright copyright) {
		this.copyright = copyright;
	}


	/**
	 * <p>
	 * Returns the copyright.
	 * </p>
	 *
	 * @return The copyright or <code>null</code>, if no copyright is set.
	 */
	public Copyright getCopyright() {
		return this.copyright;
	}
	
	
	@Override
	public void setChangeLog(ChangeLog changeLog) {
		this.changeLog = changeLog;
	}
	
	
	/**
	 * <p>
	 * Returns the change log.
	 * </p>
	 *
	 * @return The change log or <code>null</code>, if no change log is set.
	 */
	public ChangeLog getChangeLog() {
		return this.changeLog;
	}
	
	
	@Override
	public void buildDebianPackage(DataTarget target, Context context) throws IOException, BuildException {
		if (target == null)
			throw new IllegalArgumentException("Argument target is null.");
		if (context == null)
			throw new IllegalArgumentException("Argument context is null.");
		
		// --- Do nothing. ---
	}


	/* **********************************************************************
	 * **********************************************************************
	 * **********************************************************************
	 * **********************************************************************
	 * **********************************************************************
	 */


	/**
	 * <p>
	 * A node containing the data added to the builder.
	 * </p>
	 *
	 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
	 * @version <b>1.0</b>, 20.05.2016 by Gerrit Hohl
	 */
	public class DataNode implements Comparable<DataNode> {
		
		
		/** The source of a regular file. */
		private DataSource	source;
		/** The path. */
		private String		path;
		/** The target of a symbolic link. */
		private String		target;
		/** The owner. */
		private FileOwner	owner;
		/** The mode. */
		private FileMode	mode;
		
		
		/**
		 * <p>
		 * Creates a node for a directory.
		 * </p>
		 *
		 * @param path
		 *            The path.
		 * @param owner
		 *            The owner.
		 * @param mode
		 *            The mode.
		 */
		private DataNode(String path, FileOwner owner, FileMode mode) {
			super();
			
			this.source = null;
			this.path = path;
			this.target = null;
			this.owner = owner;
			this.mode = mode;
		}
		
		
		/**
		 * <p>
		 * Creates a node for a regular file.
		 * </p>
		 *
		 * @param source
		 *            The source.
		 * @param path
		 *            The path.
		 * @param owner
		 *            The owner.
		 * @param mode
		 *            The mode.
		 */
		private DataNode(DataSource source, String path, FileOwner owner, FileMode mode) {
			super();

			this.source = source;
			this.path = path;
			this.target = null;
			this.owner = owner;
			this.mode = mode;
		}
		
		
		/**
		 * <p>
		 * Creates a node for a symbolic link.
		 * </p>
		 *
		 * @param path
		 *            The path.
		 * @param target
		 *            The target.
		 * @param owner
		 *            The owner.
		 * @param mode
		 *            The mode.
		 */
		private DataNode(String path, String target, FileOwner owner, FileMode mode) {
			super();

			this.source = null;
			this.path = path;
			this.target = target;
			this.owner = owner;
			this.mode = mode;
		}
		
		
		/**
		 * <p>
		 * Returns the source of a regular file.
		 * </p>
		 *
		 * @return The source or <code>null</code>, if the node doesn't
		 *         represent a regular file.
		 * @see #isFile()
		 */
		public DataSource getSource() {
			return this.source;
		}
		
		
		/**
		 * <p>
		 * Returns the path.
		 * </p>
		 *
		 * @return The path.
		 */
		public String getPath() {
			return this.path;
		}
		
		
		/**
		 * <p>
		 * Returns the target of a symbolic link.
		 * </p>
		 *
		 * @return The target or <code>null</code>, if the node doesn't
		 *         represent a symbolic link.
		 * @see #isSymLink()
		 */
		public String getTarget() {
			return this.target;
		}
		
		
		/**
		 * <p>
		 * Returns the owner.
		 * </p>
		 *
		 * @return The owner.
		 */
		public FileOwner getOwner() {
			return this.owner;
		}
		
		
		/**
		 * <p>
		 * Returns the mode.
		 * </p>
		 *
		 * @return The mode.
		 */
		public FileMode getMode() {
			return this.mode;
		}
		
		
		/**
		 * <p>
		 * Returns the flag if the node represents a directory.
		 * </p>
		 *
		 * @return The flag.
		 */
		public boolean isDirectory() {
			return ((this.source == null) && (this.target == null));
		}


		/**
		 * <p>
		 * Returns the flag if the node represents a regular file.
		 * </p>
		 * <p>
		 * Symbolic links are also handled as regular files.
		 * </p>
		 *
		 * @return The flag.
		 * @see #isSymLink()
		 */
		public boolean isFile() {
			return ((this.source != null) || (this.target != null));
		}


		/**
		 * <p>
		 * Returns the flag if the node represents a symbolic link.
		 * </p>
		 *
		 * @return The flag.
		 * @see #getTarget()
		 */
		public boolean isSymLink() {
			return (this.target != null);
		}


		@Override
		public int compareTo(DataNode o) {
			return this.path.compareTo(o.path);
		}
		
		
	}
	
	
}
