/*
 * dpkg - Debian Package library and the Debian Package Maven plugin
 * (c) Copyright 2016 Gerrit Hohl
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package net.sourceforge.javadpkg.impl;

import java.io.IOException;
import java.util.Arrays;
import java.util.List;
import java.util.Map;

import org.junit.Assert;
import org.junit.Test;

import net.sourceforge.javadpkg.AbstractDpkgTest;
import net.sourceforge.javadpkg.Context;
import net.sourceforge.javadpkg.Copyright;
import net.sourceforge.javadpkg.CopyrightLicense;
import net.sourceforge.javadpkg.CopyrightParser;
import net.sourceforge.javadpkg.FilesCopyright;
import net.sourceforge.javadpkg.ParseException;
import net.sourceforge.javadpkg.Warning;
import net.sourceforge.javadpkg.impl.ContextImpl;
import net.sourceforge.javadpkg.impl.CopyrightParserImpl;
import net.sourceforge.javadpkg.io.DataSource;


/**
 * <p>
 * Performs some tests on the {@link CopyrightParserImpl} class.
 * </p>
 *
 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
 * @version <b>1.0</b>, 04.05.2016 by Gerrit Hohl
 */
public class CopyrightParserImplTest extends AbstractDpkgTest {
	
	
	/**
	 * <p>
	 * Performs some tests on the {@link CopyrightParserImpl} class.
	 * </p>
	 */
	@SuppressWarnings("resource")
	@Test
	public void test() {
		CopyrightParser parser;
		DataSource source;
		Context context;
		Copyright copyright;
		List<Warning> warnings;
		Warning warning;
		List<FilesCopyright> filesCopyrights;
		FilesCopyright filesCopyright;
		List<String> patterns;
		Map<String, CopyrightLicense> licenses;
		CopyrightLicense license;


		parser = new CopyrightParserImpl();
		
		
		// --- Parameters ---
		source = createSource(Arrays.asList(""), "copyright");
		context = new ContextImpl();
		try {
			parser.parseCopyright(null, context);
			Assert.fail("Expected an exception, but none was thrown.");
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		} catch (IOException | ParseException e) {
			e.printStackTrace();
			Assert.fail("An unexpected exception was thrown: " + e.getMessage());
			return;
		}
		try {
			parser.parseCopyright(source, null);
			Assert.fail("Expected an exception, but none was thrown.");
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		} catch (IOException | ParseException e) {
			e.printStackTrace();
			Assert.fail("An unexpected exception was thrown: " + e.getMessage());
			return;
		}
		
		
		// --- Parse successfully (simple) ---
		source = createSource(Arrays.asList("Format: http://www.debian.org/doc/packaging-manuals/copyright-format/1.0/",
				"Upstream-Name: SOFTware", "Upstream-Contact: John Doe <john.doe@example.com>",
				"Source: http://www.example.com/software/project"), "copyright");
		try {
			copyright = parser.parseCopyright(source, context);
		} catch (IOException | ParseException e) {
			e.printStackTrace();
			Assert.fail("An unexpected exception was thrown: " + e.getMessage());
			return;
		}
		warnings = context.getWarnings();
		Assert.assertNotNull(warnings);
		Assert.assertEquals(0, warnings.size());
		Assert.assertNotNull(copyright);
		Assert.assertEquals("http://www.debian.org/doc/packaging-manuals/copyright-format/1.0/", copyright.getFormat());
		Assert.assertEquals("SOFTware", copyright.getUpstreamName());
		Assert.assertEquals("John Doe <john.doe@example.com>", copyright.getUpstreamContact());
		Assert.assertEquals("http://www.example.com/software/project", copyright.getSource());
		Assert.assertNull(copyright.getDisclaimer());
		Assert.assertNull(copyright.getComment());
		Assert.assertNull(copyright.getLicense());
		Assert.assertNull(copyright.getCopyright());
		filesCopyrights = copyright.getFilesCopyrights();
		Assert.assertNotNull(filesCopyrights);
		Assert.assertEquals(0, filesCopyrights.size());
		licenses = copyright.getLicenses();
		Assert.assertNotNull(licenses);
		Assert.assertEquals(0, licenses.size());
		
		
		// --- Parse successfully (complex) ---
		source = createSource(Arrays.asList("Format: http://www.debian.org/doc/packaging-manuals/copyright-format/1.0/",
				"Upstream-Name: SOFTware", "Upstream-Contact: John Doe <john.doe@example.com>",
				"Source: http://www.example.com/software/project", "Disclaimer: disclaimer", "Comment: comment",
				"License: MPL-1.1 or GPL-2 or LGPL-2.1", "Copyright: 1993, John Doe", " 1993, Joe Average", "",
				"Files: src/js/editline/*", "Copyright: 1993, John Doe", " 1993, Joe Average", "", "Files: src/js/fdlibm/*",
				"Copyright: 1993, J-Random Corporation", "License: MPL-1.1", "", "Files: src/js/liba/* src/js/libb/*",
				"Copyright: 2014, A-Corporation", "", "License: MPL-1.1", " [MPL-1.1-LICENSE-TEXT]", "", "License: GPL-2",
				" [GPL-2-LICENSE-TEXT]", "", "License: LGPL-2.1", " [LGPL-2.1-LICENSE-TEXT]"), "copyright");
		try {
			copyright = parser.parseCopyright(source, context);
		} catch (IOException | ParseException e) {
			e.printStackTrace();
			Assert.fail("An unexpected exception was thrown: " + e.getMessage());
			return;
		}
		warnings = context.getWarnings();
		Assert.assertNotNull(warnings);
		Assert.assertEquals(0, warnings.size());
		Assert.assertNotNull(copyright);
		Assert.assertEquals("http://www.debian.org/doc/packaging-manuals/copyright-format/1.0/", copyright.getFormat());
		Assert.assertEquals("SOFTware", copyright.getUpstreamName());
		Assert.assertEquals("John Doe <john.doe@example.com>", copyright.getUpstreamContact());
		Assert.assertEquals("http://www.example.com/software/project", copyright.getSource());
		Assert.assertEquals("disclaimer", copyright.getDisclaimer());
		Assert.assertEquals("comment", copyright.getComment());
		license = copyright.getLicense();
		Assert.assertNotNull(license);
		Assert.assertEquals("MPL-1.1 or GPL-2 or LGPL-2.1", license.getName());
		Assert.assertNull(license.getText());
		Assert.assertNull(license.getComment());
		Assert.assertEquals("1993, John Doe\n1993, Joe Average", copyright.getCopyright());
		filesCopyrights = copyright.getFilesCopyrights();
		Assert.assertNotNull(filesCopyrights);
		Assert.assertEquals(3, filesCopyrights.size());

		filesCopyright = filesCopyrights.get(0);
		Assert.assertNotNull(filesCopyright);
		patterns = filesCopyright.getFiles();
		Assert.assertNotNull(patterns);
		Assert.assertEquals(1, patterns.size());
		Assert.assertEquals("src/js/editline/*", patterns.get(0));
		Assert.assertEquals("1993, John Doe\n1993, Joe Average", filesCopyright.getCopyright());
		Assert.assertNull(filesCopyright.getLicense());
		Assert.assertNull(filesCopyright.getComment());

		filesCopyright = filesCopyrights.get(1);
		Assert.assertNotNull(filesCopyright);
		patterns = filesCopyright.getFiles();
		Assert.assertNotNull(patterns);
		Assert.assertEquals(1, patterns.size());
		Assert.assertEquals("src/js/fdlibm/*", patterns.get(0));
		Assert.assertEquals("1993, J-Random Corporation", filesCopyright.getCopyright());
		license = filesCopyright.getLicense();
		Assert.assertNotNull(license);
		Assert.assertEquals("MPL-1.1", license.getName());
		Assert.assertNull(license.getText());
		Assert.assertNull(license.getComment());
		Assert.assertNull(filesCopyright.getComment());

		filesCopyright = filesCopyrights.get(2);
		Assert.assertNotNull(filesCopyright);
		patterns = filesCopyright.getFiles();
		Assert.assertNotNull(patterns);
		Assert.assertEquals(2, patterns.size());
		Assert.assertEquals("src/js/liba/*", patterns.get(0));
		Assert.assertEquals("src/js/libb/*", patterns.get(1));
		Assert.assertEquals("2014, A-Corporation", filesCopyright.getCopyright());
		Assert.assertNull(filesCopyright.getLicense());
		Assert.assertNull(filesCopyright.getComment());

		licenses = copyright.getLicenses();
		Assert.assertNotNull(licenses);
		Assert.assertEquals(3, licenses.size());

		license = licenses.get("MPL-1.1");
		Assert.assertNotNull(license);
		Assert.assertEquals("MPL-1.1", license.getName());
		Assert.assertEquals("[MPL-1.1-LICENSE-TEXT]", license.getText());
		Assert.assertNull(license.getComment());

		license = licenses.get("GPL-2");
		Assert.assertNotNull(license);
		Assert.assertEquals("GPL-2", license.getName());
		Assert.assertEquals("[GPL-2-LICENSE-TEXT]", license.getText());
		Assert.assertNull(license.getComment());

		license = licenses.get("LGPL-2.1");
		Assert.assertNotNull(license);
		Assert.assertEquals("LGPL-2.1", license.getName());
		Assert.assertEquals("[LGPL-2.1-LICENSE-TEXT]", license.getText());
		Assert.assertNull(license.getComment());


		// --- Parse successfully (dkms_2.2.0.3-1.1ubuntu5.14.04.1_all.deb) ---
		source = createSource(Arrays.asList("Source: http://linux.dell.com/dkms/", "", "Files: debian/HOWTO.Debian",
				"Copyright: © 2008, Dell, Inc. <dkms-devel@lists.us.dell.com>",
				"                   Mario Limonciello <Mario_Limonciello@dell.com>", "License: GPL-2+",
				" This package is free software; you can redistribute it and/or modify",
				" it under the terms of the GNU General Public License as published by",
				" the Free Software Foundation; either version 2 of the License, or", " (at your option) any later version.",
				"", "Files: debian/*", "Copyright: © 2008-2009, David Paleino <d.paleino@gmail.com>",
				"           © 2005-2008, Matt Domsch <Matt_Domsch@dell.com>",
				"           © 2008-2009, Mario Limonciello <Mario_Limonciello@dell.com>",
				"           © 2008-2009, Giuseppe Iuculano <giuseppe@iuculano.it>", "License: GPL-2+", "", "Files: *",
				"Copyright: © 2003-2009, Dell, Inc. <dkms-devel@lists.us.dell.com>", "License: GPL-2+", "",
				"On Debian systems, the complete text of the GNU General",
				"Public License v2 can be found in `/usr/share/common-licenses/GPL-2'."), "copyright");
		try {
			copyright = parser.parseCopyright(source, context);
		} catch (IOException | ParseException e) {
			e.printStackTrace();
			Assert.fail("An unexpected exception was thrown: " + e.getMessage());
			return;
		}
		warnings = context.getWarnings();
		Assert.assertNotNull(warnings);
		Assert.assertEquals(1, warnings.size());
		warning = warnings.get(0);
		Assert.assertEquals("The copyright has an unsupported format.", warning.getText());
		Assert.assertNotNull(copyright);
		Assert.assertNull(copyright.getFormat());
		Assert.assertNull(copyright.getUpstreamName());
		Assert.assertNull(copyright.getUpstreamContact());
		Assert.assertNull(copyright.getSource());
		Assert.assertNull(copyright.getDisclaimer());
		Assert.assertNull(copyright.getComment());
		Assert.assertNull(copyright.getLicense());
		Assert.assertNull(copyright.getCopyright());
		filesCopyrights = copyright.getFilesCopyrights();
		Assert.assertNotNull(filesCopyrights);
		Assert.assertEquals(0, filesCopyrights.size());
		licenses = copyright.getLicenses();
		Assert.assertNotNull(licenses);
		Assert.assertEquals(0, licenses.size());
	}
	
	
}
