/*
 * dpkg - Debian Package library and the Debian Package Maven plugin
 * (c) Copyright 2016 Gerrit Hohl
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package net.sourceforge.javadpkg.impl;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.Arrays;
import java.util.List;

import org.junit.Assert;
import org.junit.Test;

import net.sourceforge.javadpkg.AbstractDpkgTest;
import net.sourceforge.javadpkg.BuildException;
import net.sourceforge.javadpkg.Context;
import net.sourceforge.javadpkg.CopyrightBuilder;
import net.sourceforge.javadpkg.Warning;
import net.sourceforge.javadpkg.impl.ContextImpl;
import net.sourceforge.javadpkg.impl.CopyrightBuilderImpl;
import net.sourceforge.javadpkg.impl.CopyrightImpl;
import net.sourceforge.javadpkg.impl.CopyrightLicenseImpl;
import net.sourceforge.javadpkg.impl.FilesCopyrightImpl;
import net.sourceforge.javadpkg.io.DataTarget;
import net.sourceforge.javadpkg.io.impl.DataStreamTarget;


/**
 * <p>
 * Performs some tests on the {@link CopyrightBuilderImpl} class.
 * </p>
 *
 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
 * @version <b>1.0</b>, 09.05.2016 by Gerrit Hohl
 */
public class CopyrightBuilderImplTest extends AbstractDpkgTest {
	
	
	/**
	 * <p>
	 * Performs some tests on the {@link CopyrightBuilderImpl} class.
	 * </p>
	 */
	@SuppressWarnings("resource")
	@Test
	public void test() {
		CopyrightBuilder builder;
		CopyrightImpl copyright;
		Context context;
		ByteArrayOutputStream out;
		DataTarget target;
		List<Warning> warnings;
		CopyrightLicenseImpl license;
		FilesCopyrightImpl filesCopyright;
		
		
		builder = new CopyrightBuilderImpl();
		copyright = new CopyrightImpl();
		context = new ContextImpl();
		out = new ByteArrayOutputStream();
		target = new DataStreamTarget(out, "copyright", false);
		
		
		// --- Parameters ---
		try {
			builder.buildCopyright(null, target, context);
			Assert.fail("Expected an exception, but none was thrown.");
			return;
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		} catch (IOException | BuildException e) {
			e.printStackTrace();
			Assert.fail("An unexpected exception was thrown: " + e.getMessage());
			return;
		}
		try {
			builder.buildCopyright(copyright, null, context);
			Assert.fail("Expected an exception, but none was thrown.");
			return;
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		} catch (IOException | BuildException e) {
			e.printStackTrace();
			Assert.fail("An unexpected exception was thrown: " + e.getMessage());
			return;
		}
		try {
			builder.buildCopyright(copyright, target, null);
			Assert.fail("Expected an exception, but none was thrown.");
			return;
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		} catch (IOException | BuildException e) {
			e.printStackTrace();
			Assert.fail("An unexpected exception was thrown: " + e.getMessage());
			return;
		}
		
		
		// --- Build successfully (simple) ---
		out = new ByteArrayOutputStream();
		target = new DataStreamTarget(out, "copyright", false);
		copyright.setFormat("http://www.debian.org/doc/packaging-manuals/copyright-format/1.0/");
		copyright.setUpstreamName("SOFTware");
		copyright.setUpstreamContact("John Doe <john.doe@example.com>");
		copyright.setSource("http://www.example.com/software/project");
		try {
			builder.buildCopyright(copyright, target, context);
		} catch (IOException | BuildException e) {
			e.printStackTrace();
			Assert.fail("An unexpected exception was thrown: " + e.getMessage());
			return;
		}
		warnings = context.getWarnings();
		Assert.assertNotNull(warnings);
		Assert.assertEquals(0, warnings.size());
		Assert.assertEquals("Format: http://www.debian.org/doc/packaging-manuals/copyright-format/1.0/\n"
				+ "Upstream-Name: SOFTware\n" + "Upstream-Contact: John Doe <john.doe@example.com>\n"
				+ "Source: http://www.example.com/software/project\n", new String(out.toByteArray()));
		
		
		// --- Build successfully (complex) ---
		out = new ByteArrayOutputStream();
		target = new DataStreamTarget(out, "copyright", false);
		copyright.setFormat("http://www.debian.org/doc/packaging-manuals/copyright-format/1.0/");
		copyright.setUpstreamName("SOFTware");
		copyright.setUpstreamContact("John Doe <john.doe@example.com>");
		copyright.setSource("http://www.example.com/software/project");
		copyright.setDisclaimer("disclaimer");
		copyright.setComment("comment");
		license = new CopyrightLicenseImpl();
		license.setName("MPL-1.1 or GPL-2 or LGPL-2.1");
		copyright.setLicense(license);
		copyright.setCopyright("1993, John Doe\n1993, Joe Average");
		
		filesCopyright = new FilesCopyrightImpl();
		filesCopyright.setFiles(Arrays.asList("src/js/editline/*"));
		filesCopyright.setCopyright("1993, John Doe\n1993, Joe Average");
		copyright.addFilesCopyright(filesCopyright);
		
		filesCopyright = new FilesCopyrightImpl();
		filesCopyright.setFiles(Arrays.asList("src/js/fdlibm/*"));
		filesCopyright.setCopyright("1993, J-Random Corporation");
		license = new CopyrightLicenseImpl();
		license.setName("MPL-1.1");
		filesCopyright.setLicense(license);
		copyright.addFilesCopyright(filesCopyright);
		
		filesCopyright = new FilesCopyrightImpl();
		filesCopyright.setFiles(Arrays.asList("src/js/liba/* src/js/libb/*"));
		filesCopyright.setCopyright("2014, A-Corporation");
		copyright.addFilesCopyright(filesCopyright);
		
		license = new CopyrightLicenseImpl();
		license.setName("MPL-1.1");
		license.setText("[MPL-1.1-LICENSE-TEXT]");
		copyright.addLicense(license);
		
		license = new CopyrightLicenseImpl();
		license.setName("GPL-2");
		license.setText("[GPL-2-LICENSE-TEXT]");
		copyright.addLicense(license);
		
		license = new CopyrightLicenseImpl();
		license.setName("LGPL-2.1");
		license.setText("[LGPL-2.1-LICENSE-TEXT]");
		copyright.addLicense(license);
		try {
			builder.buildCopyright(copyright, target, context);
		} catch (IOException | BuildException e) {
			e.printStackTrace();
			Assert.fail("An unexpected exception was thrown: " + e.getMessage());
			return;
		}
		warnings = context.getWarnings();
		Assert.assertNotNull(warnings);
		Assert.assertEquals(0, warnings.size());
		Assert.assertEquals("Format: http://www.debian.org/doc/packaging-manuals/copyright-format/1.0/\n"
				+ "Upstream-Name: SOFTware\n" + "Upstream-Contact: John Doe <john.doe@example.com>\n"
				+ "Source: http://www.example.com/software/project\n" + "Disclaimer: disclaimer\n" + "Comment: comment\n"
				+ "License: MPL-1.1 or GPL-2 or LGPL-2.1\n" + "Copyright: 1993, John Doe\n" + " 1993, Joe Average\n" + "\n"
				+ "Files: src/js/editline/*\n" + "Copyright: 1993, John Doe\n" + " 1993, Joe Average\n" + "\n"
				+ "Files: src/js/fdlibm/*\n" + "Copyright: 1993, J-Random Corporation\n" + "License: MPL-1.1\n" + "\n"
				+ "Files: src/js/liba/* src/js/libb/*\n" + "Copyright: 2014, A-Corporation\n" + "\n" + "License: MPL-1.1\n"
				+ " [MPL-1.1-LICENSE-TEXT]\n" + "\n" + "License: GPL-2\n" + " [GPL-2-LICENSE-TEXT]\n" + "\n"
				+ "License: LGPL-2.1\n" + " [LGPL-2.1-LICENSE-TEXT]\n", new String(out.toByteArray()));
	}


}
