/*
 * dpkg - Debian Package library and the Debian Package Maven plugin
 * (c) Copyright 2016 Gerrit Hohl
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package net.sourceforge.javadpkg.impl;

import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.Date;
import java.util.List;
import java.util.Locale;

import org.junit.Assert;
import org.junit.Test;

import net.sourceforge.javadpkg.AbstractDpkgTest;
import net.sourceforge.javadpkg.ChangeLog;
import net.sourceforge.javadpkg.ChangeLogParser;
import net.sourceforge.javadpkg.ChangeLogVersionEntry;
import net.sourceforge.javadpkg.ChangeLogVersionEntryDetail;
import net.sourceforge.javadpkg.Context;
import net.sourceforge.javadpkg.ParseException;
import net.sourceforge.javadpkg.Warning;
import net.sourceforge.javadpkg.impl.ChangeLogParserImpl;
import net.sourceforge.javadpkg.impl.ContextImpl;
import net.sourceforge.javadpkg.io.DataSource;


/**
 * <p>
 * Performs some tests on the {@link ChangeLogParserImpl} class.
 * </p>
 *
 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
 * @version <b>1.0</b>, 04.05.2016 by Gerrit Hohl
 */
public class ChangeLogParserImplTest extends AbstractDpkgTest {
	
	
	/** The format of the timestamp. */
	private static final SimpleDateFormat TIMESTAMP_FORMAT = new SimpleDateFormat("EEE, dd MMM yyyy HH:mm:ss Z",
			Locale.ENGLISH);


	/**
	 * <p>
	 * Performs some tests on the
	 * {@link ChangeLogParserImpl#parseChangeLog(DataSource, Context)} method.
	 * </p>
	 */
	@SuppressWarnings("resource")
	@Test
	public void testParseChangeLog() {
		ChangeLogParser parser;
		DataSource source;
		Context context;
		ChangeLog changeLog;
		List<Warning> warnings;
		List<ChangeLogVersionEntry> entries;
		ChangeLogVersionEntry entry;
		List<String> distributions;
		List<ChangeLogVersionEntryDetail> details;
		ChangeLogVersionEntryDetail detail;


		parser = new ChangeLogParserImpl();
		
		
		// --- Parameters ---
		source = createSource(Arrays.asList(""), "copyright");
		context = new ContextImpl();
		try {
			parser.parseChangeLog(null, context);
			Assert.fail("Expected an exception, but none was thrown.");
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		} catch (IOException | ParseException e) {
			e.printStackTrace();
			Assert.fail("An unexpected exception was thrown: " + e.getMessage());
			return;
		}
		try {
			parser.parseChangeLog(source, null);
			Assert.fail("Expected an exception, but none was thrown.");
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		} catch (IOException | ParseException e) {
			e.printStackTrace();
			Assert.fail("An unexpected exception was thrown: " + e.getMessage());
			return;
		}
		
		
		// --- Parse successfully (simple) ---
		source = createSource(Arrays.asList("mypackage (1.0.0) lucid precise trusty; urgency=low", "", "  * change details",
				"    more change details", "    .", "    more change details after an empty line", "",
				"  * even more change details", "", " -- John Doe <j.doe@example.com>  Wed, 04 May 2016 15:05:00 +0200"),
				"copyright");
		try {
			changeLog = parser.parseChangeLog(source, context);
		} catch (IOException | ParseException e) {
			e.printStackTrace();
			Assert.fail("An unexpected exception was thrown: " + e.getMessage());
			return;
		}
		warnings = context.getWarnings();
		Assert.assertNotNull(warnings);
		Assert.assertEquals(0, warnings.size());
		Assert.assertNotNull(changeLog);
		entries = changeLog.getEntries();
		Assert.assertNotNull(entries);
		Assert.assertEquals(1, entries.size());
		entry = entries.get(0);
		Assert.assertNotNull(entry);
		Assert.assertNotNull(entry.getPackage());
		Assert.assertEquals("mypackage", entry.getPackage().getName());
		Assert.assertNotNull(entry.getVersion());
		Assert.assertEquals("1.0.0", entry.getVersion().getText());
		distributions = entry.getDistributions();
		Assert.assertNotNull(distributions);
		Assert.assertEquals(3, distributions.size());
		Assert.assertEquals("lucid", distributions.get(0));
		Assert.assertEquals("precise", distributions.get(1));
		Assert.assertEquals("trusty", distributions.get(2));
		Assert.assertNotNull(entry.getUrgency());
		Assert.assertEquals("low", entry.getUrgency().getText());
		details = entry.getDetails();
		Assert.assertNotNull(details);
		Assert.assertEquals(2, details.size());
		detail = details.get(0);
		Assert.assertNotNull(detail);
		Assert.assertEquals("change details\nmore change details\n\nmore change details after an empty line", detail.getText());
		detail = details.get(1);
		Assert.assertNotNull(detail);
		Assert.assertEquals("even more change details", detail.getText());
		Assert.assertNotNull(entry.getMaintainer());
		Assert.assertEquals("John Doe", entry.getMaintainer().getName());
		Assert.assertEquals("j.doe@example.com", entry.getMaintainer().getAddress());
		Assert.assertNotNull(entry.getDate());
		Assert.assertEquals(this.parseDate("Wed, 04 May 2016 15:05:00 +0200"), entry.getDate());
		
		
		// --- Parse successfully (complex) ---
		source = createSource(
				Arrays.asList("mypackage (1.1.0) trusty; urgency=low", "", "  * change details", "    more change details", "",
						"  * even more change details", "", " -- John Doe <j.doe@example.com>  Wed, 04 May 2016 15:05:00 +0200",
						"", "mypackage (1.0.0) lucid; urgency=HIGH", "  * change details", "", "",
						"  * even more change details", " -- John Doe <j.doe@example.com>  Fri, 02 Jan 2015 10:37:00 +0100"),
				"copyright");
		try {
			changeLog = parser.parseChangeLog(source, context);
		} catch (IOException | ParseException e) {
			e.printStackTrace();
			Assert.fail("An unexpected exception was thrown: " + e.getMessage());
			return;
		}
		warnings = context.getWarnings();
		Assert.assertNotNull(warnings);
		Assert.assertEquals(0, warnings.size());
		Assert.assertNotNull(changeLog);
		entries = changeLog.getEntries();
		Assert.assertNotNull(entries);
		Assert.assertEquals(2, entries.size());

		entry = entries.get(0);
		Assert.assertNotNull(entry);
		Assert.assertNotNull(entry.getPackage());
		Assert.assertEquals("mypackage", entry.getPackage().getName());
		Assert.assertNotNull(entry.getVersion());
		Assert.assertEquals("1.1.0", entry.getVersion().getText());
		distributions = entry.getDistributions();
		Assert.assertNotNull(distributions);
		Assert.assertEquals(1, distributions.size());
		Assert.assertEquals("trusty", distributions.get(0));
		Assert.assertNotNull(entry.getUrgency());
		Assert.assertEquals("low", entry.getUrgency().getText());
		details = entry.getDetails();
		Assert.assertNotNull(details);
		Assert.assertEquals(2, details.size());
		detail = details.get(0);
		Assert.assertNotNull(detail);
		Assert.assertEquals("change details\nmore change details", detail.getText());
		detail = details.get(1);
		Assert.assertNotNull(detail);
		Assert.assertEquals("even more change details", detail.getText());
		Assert.assertNotNull(entry.getMaintainer());
		Assert.assertEquals("John Doe", entry.getMaintainer().getName());
		Assert.assertEquals("j.doe@example.com", entry.getMaintainer().getAddress());
		Assert.assertNotNull(entry.getDate());
		Assert.assertEquals(this.parseDate("Wed, 04 May 2016 15:05:00 +0200"), entry.getDate());

		entry = entries.get(1);
		Assert.assertNotNull(entry);
		Assert.assertNotNull(entry.getPackage());
		Assert.assertEquals("mypackage", entry.getPackage().getName());
		Assert.assertNotNull(entry.getVersion());
		Assert.assertEquals("1.0.0", entry.getVersion().getText());
		distributions = entry.getDistributions();
		Assert.assertNotNull(distributions);
		Assert.assertEquals(1, distributions.size());
		Assert.assertEquals("lucid", distributions.get(0));
		Assert.assertNotNull(entry.getUrgency());
		Assert.assertEquals("high", entry.getUrgency().getText());
		details = entry.getDetails();
		Assert.assertNotNull(details);
		Assert.assertEquals(2, details.size());
		detail = details.get(0);
		Assert.assertNotNull(detail);
		Assert.assertEquals("change details", detail.getText());
		detail = details.get(1);
		Assert.assertNotNull(detail);
		Assert.assertEquals("even more change details", detail.getText());
		Assert.assertNotNull(entry.getMaintainer());
		Assert.assertEquals("John Doe", entry.getMaintainer().getName());
		Assert.assertEquals("j.doe@example.com", entry.getMaintainer().getAddress());
		Assert.assertNotNull(entry.getDate());
		Assert.assertEquals(this.parseDate("Fri, 02 Jan 2015 10:37:00 +0100"), entry.getDate());
	}
	
	
	/**
	 * <p>
	 * Parses a timestamp.
	 * </p>
	 *
	 * @param value
	 *            The value.
	 * @return The timestamp.
	 */
	private Date parseDate(String value) {
		Date date;


		try {
			date = TIMESTAMP_FORMAT.parse(value);
		} catch (java.text.ParseException e) {
			e.printStackTrace();
			Assert.fail("Couldn't prepare timestamp |" + value + "|: " + e.getMessage());
			return null;
		}
		return date;
	}


}
