/*
 * dpkg - Debian Package library and the Debian Package Maven plugin
 * (c) Copyright 2016 Gerrit Hohl
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package net.sourceforge.javadpkg.impl;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.Date;
import java.util.List;
import java.util.Locale;

import org.junit.Assert;
import org.junit.Test;

import net.sourceforge.javadpkg.AbstractDpkgTest;
import net.sourceforge.javadpkg.BuildException;
import net.sourceforge.javadpkg.ChangeLogBuilder;
import net.sourceforge.javadpkg.Context;
import net.sourceforge.javadpkg.Warning;
import net.sourceforge.javadpkg.impl.ChangeLogBuilderImpl;
import net.sourceforge.javadpkg.impl.ChangeLogImpl;
import net.sourceforge.javadpkg.impl.ChangeLogVersionEntryDetailImpl;
import net.sourceforge.javadpkg.impl.ChangeLogVersionEntryImpl;
import net.sourceforge.javadpkg.impl.ContextImpl;
import net.sourceforge.javadpkg.io.DataTarget;
import net.sourceforge.javadpkg.io.impl.DataStreamTarget;


/**
 * <p>
 * Performs some tests on the {@link ChangeLogBuilderImpl} class.
 * </p>
 *
 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
 * @version <b>1.0</b>, 09.05.2016 by Gerrit Hohl
 */
public class ChangeLogBuilderImplTest extends AbstractDpkgTest {
	
	
	/** The format of the timestamp. */
	private static final SimpleDateFormat TIMESTAMP_FORMAT = new SimpleDateFormat("EEE, dd MMM yyyy HH:mm:ss Z",
			Locale.ENGLISH);
	
	
	/**
	 * <p>
	 * Performs some tests on the {@link ChangeLogBuilderImpl} class.
	 * </p>
	 */
	@SuppressWarnings("resource")
	@Test
	public void test() {
		ChangeLogBuilder builder;
		ChangeLogImpl changeLog;
		ByteArrayOutputStream out;
		DataTarget target;
		Context context;
		List<Warning> warnings;
		ChangeLogVersionEntryImpl entry;
		
		
		builder = new ChangeLogBuilderImpl();
		changeLog = new ChangeLogImpl();
		out = new ByteArrayOutputStream();
		target = new DataStreamTarget(out, "changelog", false);
		context = new ContextImpl();
		
		
		// --- Parameters ---
		try {
			builder.buildChangeLog(null, target, context);
			Assert.fail("Expected an exception, but none was thrown.");
			return;
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		} catch (IOException | BuildException e) {
			e.printStackTrace();
			Assert.fail("An unexpected exception was thrown: " + e.getMessage());
			return;
		}
		try {
			builder.buildChangeLog(changeLog, null, context);
			Assert.fail("Expected an exception, but none was thrown.");
			return;
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		} catch (IOException | BuildException e) {
			e.printStackTrace();
			Assert.fail("An unexpected exception was thrown: " + e.getMessage());
			return;
		}
		try {
			builder.buildChangeLog(changeLog, target, null);
			Assert.fail("Expected an exception, but none was thrown.");
			return;
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		} catch (IOException | BuildException e) {
			e.printStackTrace();
			Assert.fail("An unexpected exception was thrown: " + e.getMessage());
			return;
		}
		
		
		// --- Build successfully (simple) ---
		changeLog = new ChangeLogImpl();
		entry = new ChangeLogVersionEntryImpl();
		entry.setPackageName(createPackageName("mypackage"));
		entry.setVersion(createPackageVersion("1.0.0"));
		entry.setDistributions(Arrays.asList("lucid", "precise", "trusty"));
		entry.setUrgency(createChangeLogUrgency("low"));
		entry.addDetail(new ChangeLogVersionEntryDetailImpl(
				"change details\nmore change details\n\nmore change details after an empty line"));
		entry.addDetail(new ChangeLogVersionEntryDetailImpl("even more change details"));
		entry.setMaintainer(createPackageMaintainer("John Doe <j.doe@example.com>"));
		entry.setDate(this.parseDate("Wed, 04 May 2016 15:05:00 +0200"));
		changeLog.addEntry(entry);
		out = new ByteArrayOutputStream();
		target = new DataStreamTarget(out, "changelog", false);
		try {
			builder.buildChangeLog(changeLog, target, context);
		} catch (IOException | BuildException e) {
			e.printStackTrace();
			Assert.fail("An unexpected exception was thrown: " + e.getMessage());
			return;
		}
		warnings = context.getWarnings();
		Assert.assertNotNull(warnings);
		Assert.assertEquals(0, warnings.size());
		Assert.assertEquals("mypackage (1.0.0) lucid precise trusty; urgency=low\n" + "\n" + "  * change details\n"
				+ "    more change details\n" + "    .\n" + "    more change details after an empty line\n" + "\n"
				+ "  * even more change details\n" + "\n"
				+ " -- John Doe <j.doe@example.com>  Wed, 04 May 2016 15:05:00 +0200\n", new String(out.toByteArray()));
		
		
		// --- Build successfully (complex) ---
		changeLog = new ChangeLogImpl();

		entry = new ChangeLogVersionEntryImpl();
		entry.setPackageName(createPackageName("mypackage"));
		entry.setVersion(createPackageVersion("1.1.0"));
		entry.setDistributions(Arrays.asList("trusty"));
		entry.setUrgency(createChangeLogUrgency("low"));
		entry.addDetail(new ChangeLogVersionEntryDetailImpl("change details\nmore change details"));
		entry.addDetail(new ChangeLogVersionEntryDetailImpl("even more change details"));
		entry.setMaintainer(createPackageMaintainer("John Doe <j.doe@example.com>"));
		entry.setDate(this.parseDate("Wed, 04 May 2016 15:05:00 +0200"));
		changeLog.addEntry(entry);

		entry = new ChangeLogVersionEntryImpl();
		entry.setPackageName(createPackageName("mypackage"));
		entry.setVersion(createPackageVersion("1.0.0"));
		entry.setDistributions(Arrays.asList("lucid"));
		entry.setUrgency(createChangeLogUrgency("high"));
		entry.addDetail(new ChangeLogVersionEntryDetailImpl("change details"));
		entry.addDetail(new ChangeLogVersionEntryDetailImpl("even more change details"));
		entry.setMaintainer(createPackageMaintainer("John Doe <j.doe@example.com>"));
		entry.setDate(this.parseDate("Fri, 02 Jan 2015 10:37:00 +0100"));
		changeLog.addEntry(entry);

		out = new ByteArrayOutputStream();
		target = new DataStreamTarget(out, "changelog", false);
		try {
			builder.buildChangeLog(changeLog, target, context);
		} catch (IOException | BuildException e) {
			e.printStackTrace();
			Assert.fail("An unexpected exception was thrown: " + e.getMessage());
			return;
		}
		warnings = context.getWarnings();
		Assert.assertNotNull(warnings);
		Assert.assertEquals(0, warnings.size());
		Assert.assertEquals("mypackage (1.1.0) trusty; urgency=low\n" + "\n" + "  * change details\n"
				+ "    more change details\n" + "\n" + "  * even more change details\n" + "\n"
				+ " -- John Doe <j.doe@example.com>  Wed, 04 May 2016 15:05:00 +0200\n" + "\n"
				+ "mypackage (1.0.0) lucid; urgency=high\n" + "\n" + "  * change details\n" + "\n"
				+ "  * even more change details\n" + "\n"
				+ " -- John Doe <j.doe@example.com>  Fri, 02 Jan 2015 10:37:00 +0100\n", new String(out.toByteArray()));
	}
	
	
	/**
	 * <p>
	 * Parses a timestamp.
	 * </p>
	 *
	 * @param value
	 *            The value.
	 * @return The timestamp.
	 */
	private Date parseDate(String value) {
		Date date;
		
		
		try {
			date = TIMESTAMP_FORMAT.parse(value);
		} catch (java.text.ParseException e) {
			e.printStackTrace();
			Assert.fail("Couldn't prepare timestamp |" + value + "|: " + e.getMessage());
			return null;
		}
		return date;
	}
	
	
}
