/*
 * dpkg - Debian Package library and the Debian Package Maven plugin
 * (c) Copyright 2016 Gerrit Hohl
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package net.sourceforge.javadpkg.control.impl;

import java.util.List;

import org.junit.Assert;
import org.junit.Test;

import net.sourceforge.javadpkg.AbstractDpkgTest;
import net.sourceforge.javadpkg.BuildException;
import net.sourceforge.javadpkg.Context;
import net.sourceforge.javadpkg.ParseException;
import net.sourceforge.javadpkg.Warning;
import net.sourceforge.javadpkg.control.PackageDependency;
import net.sourceforge.javadpkg.control.PackageDependencyBuilder;
import net.sourceforge.javadpkg.control.PackageDependencyParser;
import net.sourceforge.javadpkg.control.PackageNameBuilder;
import net.sourceforge.javadpkg.control.PackageNameParser;
import net.sourceforge.javadpkg.control.PackageVersionBuilder;
import net.sourceforge.javadpkg.control.PackageVersionParser;
import net.sourceforge.javadpkg.control.PackageVersionRelationOperatorBuilder;
import net.sourceforge.javadpkg.control.PackageVersionRelationOperatorParser;
import net.sourceforge.javadpkg.control.impl.PackageDependencyBuilderImpl;
import net.sourceforge.javadpkg.control.impl.PackageDependencyParserImpl;
import net.sourceforge.javadpkg.control.impl.PackageNameBuilderImpl;
import net.sourceforge.javadpkg.control.impl.PackageNameParserImpl;
import net.sourceforge.javadpkg.control.impl.PackageVersionBuilderImpl;
import net.sourceforge.javadpkg.control.impl.PackageVersionParserImpl;
import net.sourceforge.javadpkg.control.impl.PackageVersionRelationOperatorBuilderImpl;
import net.sourceforge.javadpkg.control.impl.PackageVersionRelationOperatorParserImpl;
import net.sourceforge.javadpkg.impl.ContextImpl;


/**
 * <p>
 * Performs some tests on the {@link PackageDependencyBuilderImpl} class.
 * </p>
 *
 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
 * @version <b>1.0</b>, 29.04.2016 by Gerrit Hohl
 */
public class PackageDependencyBuilderImplTest extends AbstractDpkgTest {
	
	
	/** The builder for the package name. */
	private PackageNameBuilder						packageNameBuilder;
	/** The builder for the version relation operator. */
	private PackageVersionRelationOperatorBuilder	packageVersionRelationOperatorBuilder;
	/** The builder for the version. */
	private PackageVersionBuilder					packageVersionBuilder;

	/** The parser for the package name. */
	private PackageNameParser						packageNameParser;
	/** The parser for the version relation operator. */
	private PackageVersionRelationOperatorParser	packageVersionRelationOperatorParser;
	/** The parser for the version. */
	private PackageVersionParser					packageVersionParser;
	/** The parser for the package dependency. */
	private PackageDependencyParser					packageDependencyParser;
													
													
	@Override
	public void setUp() throws Exception {
		super.setUp();
		
		this.packageNameBuilder = new PackageNameBuilderImpl();
		this.packageVersionRelationOperatorBuilder = new PackageVersionRelationOperatorBuilderImpl();
		this.packageVersionBuilder = new PackageVersionBuilderImpl();

		this.packageNameParser = new PackageNameParserImpl();
		this.packageVersionRelationOperatorParser = new PackageVersionRelationOperatorParserImpl();
		this.packageVersionParser = new PackageVersionParserImpl();
		this.packageDependencyParser = new PackageDependencyParserImpl(this.packageNameParser,
				this.packageVersionRelationOperatorParser, this.packageVersionParser);
	}
	
	
	/**
	 * <p>
	 * Performs some tests on the
	 * {@link PackageDependencyBuilderImpl#buildPackageDependency(PackageDependency, Context)}
	 * method.
	 * </p>
	 */
	@Test
	public void testBuildPackageDependency() {
		PackageDependencyBuilder builder;
		PackageDependency dependency;
		Context context;
		List<Warning> warnings;
		String value;


		builder = new PackageDependencyBuilderImpl(this.packageNameBuilder, this.packageVersionRelationOperatorBuilder,
				this.packageVersionBuilder);
		dependency = this.createPackageDependency("libc6 (>= 2.14)");
		context = new ContextImpl();
		
		
		// --- Parameters ---
		try {
			builder.buildPackageDependency(null, context);
			Assert.fail("Expected an exception, but none was thrown.");
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		} catch (BuildException e) {
			e.printStackTrace();
			Assert.fail("An unexpected exception was thrown: " + e.getMessage());
			return;
		}
		try {
			builder.buildPackageDependency(dependency, null);
			Assert.fail("Expected an exception, but none was thrown.");
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		} catch (BuildException e) {
			e.printStackTrace();
			Assert.fail("An unexpected exception was thrown: " + e.getMessage());
			return;
		}


		// --- Build successfully ---
		try {
			value = builder.buildPackageDependency(dependency, context);
		} catch (BuildException e) {
			e.printStackTrace();
			Assert.fail("An unexpected exception was thrown: " + e.getMessage());
			return;
		}
		warnings = context.getWarnings();
		Assert.assertNotNull(warnings);
		Assert.assertEquals(0, warnings.size());
		Assert.assertEquals("libc6 (>= 2.14)", value);
		
		
		// --- Build successfully (condition) ---
		dependency = this.createPackageDependency("debconf (>= 0.5) | debconf-2.0");
		try {
			value = builder.buildPackageDependency(dependency, context);
		} catch (BuildException e) {
			e.printStackTrace();
			Assert.fail("An unexpected exception was thrown: " + e.getMessage());
			return;
		}
		warnings = context.getWarnings();
		Assert.assertNotNull(warnings);
		Assert.assertEquals(0, warnings.size());
		Assert.assertEquals("debconf (>= 0.5) | debconf-2.0", value);
	}
	
	
	/**
	 * <p>
	 * Performs some tests on the
	 * {@link PackageDependencyBuilderImpl#buildPackageDependencies(List, Context)}
	 * method.
	 * </p>
	 */
	@Test
	public void testBuildPackageDependencies() {
		PackageDependencyBuilder builder;
		List<PackageDependency> dependencies;
		Context context;
		List<Warning> warnings;
		String value;


		builder = new PackageDependencyBuilderImpl(this.packageNameBuilder, this.packageVersionRelationOperatorBuilder,
				this.packageVersionBuilder);
		dependencies = this.createPackageDependencies("libc6 (>= 2.14)");
		context = new ContextImpl();
		
		
		// --- Parameters ---
		try {
			builder.buildPackageDependencies(null, context);
			Assert.fail("Expected an exception, but none was thrown.");
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		} catch (BuildException e) {
			e.printStackTrace();
			Assert.fail("An unexpected exception was thrown: " + e.getMessage());
			return;
		}
		try {
			builder.buildPackageDependencies(dependencies, null);
			Assert.fail("Expected an exception, but none was thrown.");
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		} catch (BuildException e) {
			e.printStackTrace();
			Assert.fail("An unexpected exception was thrown: " + e.getMessage());
			return;
		}


		// --- Build successfully ---
		try {
			value = builder.buildPackageDependencies(dependencies, context);
		} catch (BuildException e) {
			e.printStackTrace();
			Assert.fail("An unexpected exception was thrown: " + e.getMessage());
			return;
		}
		warnings = context.getWarnings();
		Assert.assertNotNull(warnings);
		Assert.assertEquals(0, warnings.size());
		Assert.assertEquals("libc6 (>= 2.14)", value);
		
		
		// --- Build successfully (condition) ---
		dependencies = this.createPackageDependencies("libc6 (>= 2.14), debconf (>= 0.5) | debconf-2.0");
		try {
			value = builder.buildPackageDependencies(dependencies, context);
		} catch (BuildException e) {
			e.printStackTrace();
			Assert.fail("An unexpected exception was thrown: " + e.getMessage());
			return;
		}
		warnings = context.getWarnings();
		Assert.assertNotNull(warnings);
		Assert.assertEquals(0, warnings.size());
		Assert.assertEquals("libc6 (>= 2.14), debconf (>= 0.5) | debconf-2.0", value);
	}
	
	
	/**
	 * <p>
	 * Prepares a dependency.
	 * </p>
	 *
	 * @param value
	 *            The value.
	 * @return The dependency.
	 */
	private PackageDependency createPackageDependency(String value) {
		PackageDependency dependency;
		Context context;
		List<Warning> warnings;
		
		
		context = new ContextImpl();
		try {
			dependency = this.packageDependencyParser.parsePackageDependency(value, context);
		} catch (ParseException e) {
			e.printStackTrace();
			Assert.fail("Couldn't prepare dependency |" + value + "|: An unexpected exception was thrown: " + e.getMessage());
			return null;
		}
		warnings = context.getWarnings();
		Assert.assertNotNull(warnings);
		Assert.assertEquals("Couldn't prepare dependency |" + value + "|: Got " + warnings.size() + " warning(s).", 0,
				warnings.size());
		return dependency;
	}
	
	
	/**
	 * <p>
	 * Prepares the dependencies.
	 * </p>
	 *
	 * @param value
	 *            The value.
	 * @return The dependencies.
	 */
	private List<PackageDependency> createPackageDependencies(String value) {
		List<PackageDependency> dependencies;
		Context context;
		List<Warning> warnings;
		
		
		context = new ContextImpl();
		try {
			dependencies = this.packageDependencyParser.parsePackageDependencies(value, context);
		} catch (ParseException e) {
			e.printStackTrace();
			Assert.fail("Couldn't prepare dependencies " + value + ": An unexpected exception was thrown: " + e.getMessage());
			return null;
		}
		warnings = context.getWarnings();
		Assert.assertNotNull(warnings);
		Assert.assertEquals("Couldn't prepare dependency |" + value + "|: Got " + warnings.size() + " warning(s).", 0,
				warnings.size());
		return dependencies;
	}


}
