/*
 * dpkg - Debian Package library and the Debian Package Maven plugin
 * (c) Copyright 2016 Gerrit Hohl
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package net.sourceforge.javadpkg.store;


/**
 * <p>
 * A {@link FileHash} implementation.
 * </p>
 *
 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
 * @version <b>1.0</b>, 27.04.2016 by Gerrit Hohl
 */
public class FileHashImpl implements FileHash {


	/** The name of the source. */
	private String	name;
	/** The path in the target system. */
	private String	path;
	/** The hash of the content. */
	private byte[]	hash;
	/** The hash of the content as hex number. */
	private String	hashAsHex;


	/**
	 * <p>
	 * Creates a file hash.
	 * </p>
	 *
	 * @param name
	 *            The name of the source.
	 * @param path
	 *            The path in the target system.
	 * @param hash
	 *            The hash of the content.
	 * @throws IllegalArgumentException
	 *             If any of the parameters are <code>null</code>.
	 */
	public FileHashImpl(String name, String path, byte[] hash) {
		super();
		
		if (name == null)
			throw new IllegalArgumentException("Argument name is null.");
		if (path == null)
			throw new IllegalArgumentException("Argument path is null.");
		if (hash == null)
			throw new IllegalArgumentException("Argument hash is null.");
			
		this.name = name;
		this.path = path;
		this.hash = hash.clone();
		this.hashAsHex = this.createHex(this.hash);
	}


	/**
	 * <p>
	 * Creates a hex representation of the number stored in the array.
	 * </p>
	 *
	 * @param data
	 *            The array.
	 * @return The hex representation.
	 */
	private String createHex(byte[] data) {
		StringBuilder sb;
		int number;
		
		
		sb = new StringBuilder();
		for (int index = 0; index < data.length; index++) {
			number = data[index] & 0xFF;
			if (number < 0x10) {
				sb.append('0');
			}
			sb.append(Integer.toHexString(number));
		}
		return sb.toString();
	}


	@Override
	public String getName() {
		return this.name;
	}
	
	
	@Override
	public String getPath() {
		return this.path;
	}
	
	
	@Override
	public byte[] getHash() {
		return this.hash.clone();
	}
	
	
	@Override
	public String getHashAsHex() {
		return this.hashAsHex;
	}
	
	
}
