/*
 * dpkg - Debian Package library and the Debian Package Maven plugin
 * (c) Copyright 2015 Gerrit Hohl
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package net.sourceforge.javadpkg.io.impl;

import java.io.IOException;
import java.io.InputStream;

import net.sourceforge.javadpkg.io.CloseHandler;


/**
 * <p>
 * An {@link InputStream} which can't be closed.
 * </p>
 * <p>
 * Calls of the {@link #close()} method don't have any effect on the underlying
 * stream.
 * </p>
 *
 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
 * @version <b>1.0</b>, 31.12.2015 by Gerrit Hohl
 */
public class UncloseableInputStream extends InputStream {


	/** The underlying stream. */
	private InputStream		in;
	/** The close handler (optional). */
	private CloseHandler	closeHandler;


	/**
	 * <p>
	 * Creates a stream.
	 * </p>
	 *
	 * @param in
	 *            The underlying stream.
	 * @throws IllegalArgumentException
	 *             If the underlying stream is <code>null</code>.
	 */
	public UncloseableInputStream(InputStream in) {
		super();
		
		if (in == null)
			throw new IllegalArgumentException("Argument in is null.");

		this.in = in;
		this.closeHandler = null;
	}


	/**
	 * <p>
	 * Creates a stream.
	 * </p>
	 *
	 * @param in
	 *            The underlying stream.
	 * @param closeHandler
	 *            The handler which is called when the {@link #close()} method
	 *            is called.
	 * @throws IllegalArgumentException
	 *             If any of the parameters are <code>null</code>.
	 */
	public UncloseableInputStream(InputStream in, CloseHandler closeHandler) {
		super();
		
		if (in == null)
			throw new IllegalArgumentException("Argument in is null.");
		if (closeHandler == null)
			throw new IllegalArgumentException("Argument closeHandler is null.");

		this.in = in;
		this.closeHandler = closeHandler;
	}


	@Override
	public int read() throws IOException {
		return this.in.read();
	}


	@Override
	public int read(byte[] b) throws IOException {
		return this.in.read(b);
	}


	@Override
	public int read(byte[] b, int off, int len) throws IOException {
		return this.in.read(b, off, len);
	}


	@Override
	public long skip(long n) throws IOException {
		return this.in.skip(n);
	}


	@Override
	public int available() throws IOException {
		return this.in.available();
	}


	@Override
	public void close() throws IOException {
		if (this.closeHandler != null)
			this.closeHandler.closed();
	}


	@Override
	public synchronized void mark(int readlimit) {
		this.in.mark(readlimit);
	}


	@Override
	public synchronized void reset() throws IOException {
		this.in.reset();
	}


	@Override
	public boolean markSupported() {
		return this.in.markSupported();
	}


}
