/*
 * dpkg - Debian Package library and the Debian Package Maven plugin
 * (c) Copyright 2016 Gerrit Hohl
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package net.sourceforge.javadpkg.io.impl;

import java.util.Date;

import net.sourceforge.javadpkg.io.FileMetaData;
import net.sourceforge.javadpkg.io.FileMode;
import net.sourceforge.javadpkg.io.FileOwner;

/**
 * <p>
 * A {@link FileMetaData} implementation.
 * </p>
 *
 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
 * @version <b>1.0</b>, 12.05.2016 by Gerrit Hohl
 */
public class FileMetaDataImpl implements FileMetaData {
	
	
	/** The path of the file excluding the name. */
	private String		path;
	/** The name of the file. */
	private String		name;
	/** The flag if the file is a directory. */
	private boolean		directory;
	/** The target path of the symbolic link. */
	private String		targetPath;
	/** The owner of the file. */
	private FileOwner	owner;
	/** The mode of the file. */
	private FileMode	mode;
	/** The length of the file. */
	private long		length;
	/** The timestamp of the last modification. */
	private Date		lastModifiedDate;
	
	
	/**
	 * <p>
	 * Creates the meta data of a directory.
	 * </p>
	 *
	 * @param path
	 *            The path of the file excluding the name.
	 * @param name
	 *            The name of the file.
	 * @param owner
	 *            The owner of the file.
	 * @param mode
	 *            The mode of the file.
	 * @param lastModifiedDate
	 *            The timestamp of the last modification.
	 * @throws IllegalArgumentException
	 *             If any of the parameters are <code>null</code>, the path is
	 *             empty, the name contains a slash or is empty also the path is
	 *             not &quot;/&quot;.
	 */
	public FileMetaDataImpl(String path, String name, FileOwner owner, FileMode mode, Date lastModifiedDate) {
		super();

		if (path == null)
			throw new IllegalArgumentException("Argument path is null.");
		if (path.isEmpty())
			throw new IllegalArgumentException("Argument path is empty.");
		if (name == null)
			throw new IllegalArgumentException("Argument name is null.");
		if (name.indexOf('/') != -1)
			throw new IllegalArgumentException("Argument name contains a slash: " + name);
		if (owner == null)
			throw new IllegalArgumentException("Argument owner is null.");
		if (mode == null)
			throw new IllegalArgumentException("Argument mode is null.");
		if (lastModifiedDate == null)
			throw new IllegalArgumentException("Argument lastModifiedDate is null.");
		
		this.path = this.revisePath(path);
		if (name.isEmpty() && !"/".equals(path))
			throw new IllegalArgumentException("An empty name is only valid for the root. Path: " + path);
		this.name = name;
		this.directory = true;
		this.targetPath = null;
		this.owner = owner;
		this.mode = mode;
		this.length = -1;
		this.lastModifiedDate = lastModifiedDate;
	}
	
	
	/**
	 * <p>
	 * Creates the meta data of a directory.
	 * </p>
	 *
	 * @param parent
	 *            The parent directory.
	 * @param name
	 *            The name of the file.
	 * @param owner
	 *            The owner of the file.
	 * @param mode
	 *            The mode of the file.
	 * @param lastModifiedDate
	 *            The timestamp of the last modification.
	 * @throws IllegalArgumentException
	 *             If any of the parameters are <code>null</code>, the parent is
	 *             not a directory, the name contains a slash or is empty.
	 */
	public FileMetaDataImpl(FileMetaData parent, String name, FileOwner owner, FileMode mode, Date lastModifiedDate) {
		super();

		if (parent == null)
			throw new IllegalArgumentException("Argument path is null.");
		if (!parent.isDirectory())
			throw new IllegalArgumentException("Argument parent |" + parent.getAbsolutePath() + "| is not a directory.");
		if (name == null)
			throw new IllegalArgumentException("Argument name is null.");
		if (name.isEmpty())
			throw new IllegalArgumentException("An empty name is only valid for the root. Path: " + parent.getAbsolutePath());
		if (name.indexOf('/') != -1)
			throw new IllegalArgumentException("Argument name contains a slash: " + name);
		if (owner == null)
			throw new IllegalArgumentException("Argument owner is null.");
		if (mode == null)
			throw new IllegalArgumentException("Argument mode is null.");
		if (lastModifiedDate == null)
			throw new IllegalArgumentException("Argument lastModifiedDate is null.");
		
		this.path = parent.getAbsolutePath();
		this.name = name;
		this.directory = true;
		this.targetPath = null;
		this.owner = owner;
		this.mode = mode;
		this.length = -1;
		this.lastModifiedDate = lastModifiedDate;
	}
	
	
	/**
	 * <p>
	 * Creates the meta data of a regular file.
	 * </p>
	 *
	 * @param path
	 *            The path of the file excluding the name.
	 * @param name
	 *            The name of the file.
	 * @param owner
	 *            The owner of the file.
	 * @param mode
	 *            The mode of the file.
	 * @param length
	 *            The length of the file or <code>-1</code>, if the length is
	 *            unknown.
	 * @param lastModifiedDate
	 *            The timestamp of the last modification.
	 * @throws IllegalArgumentException
	 *             If any of the parameters are <code>null</code>, the path is
	 *             empty, the name contains a slash, the name is empty or the
	 *             length is less than <code>-1</code>.
	 */
	public FileMetaDataImpl(String path, String name, FileOwner owner, FileMode mode, long length, Date lastModifiedDate) {
		super();

		if (path == null)
			throw new IllegalArgumentException("Argument path is null.");
		if (path.isEmpty())
			throw new IllegalArgumentException("Argument path is empty.");
		if (name == null)
			throw new IllegalArgumentException("Argument name is null.");
		if (name.isEmpty())
			throw new IllegalArgumentException("Argument name is empty.");
		if (name.indexOf('/') != -1)
			throw new IllegalArgumentException("Argument name contains a slash: " + name);
		if (owner == null)
			throw new IllegalArgumentException("Argument owner is null.");
		if (mode == null)
			throw new IllegalArgumentException("Argument mode is null.");
		if (length < -1)
			throw new IllegalArgumentException("Argument length is less than -1: " + length);
		if (lastModifiedDate == null)
			throw new IllegalArgumentException("Argument lastModifiedDate is null.");
		
		this.path = this.revisePath(path);
		this.name = name;
		this.directory = false;
		this.targetPath = null;
		this.owner = owner;
		this.mode = mode;
		this.length = length;
		this.lastModifiedDate = lastModifiedDate;
	}


	/**
	 * <p>
	 * Creates the meta data of a regular file.
	 * </p>
	 *
	 * @param parent
	 *            The parent directory.
	 * @param name
	 *            The name of the file.
	 * @param owner
	 *            The owner of the file.
	 * @param mode
	 *            The mode of the file.
	 * @param length
	 *            The length of the file or <code>-1</code>, if the length is
	 *            unknown.
	 * @param lastModifiedDate
	 *            The timestamp of the last modification.
	 * @throws IllegalArgumentException
	 *             If any of the parameters are <code>null</code>, the parent is
	 *             not a directory, the name contains a slash or the length is
	 *             less than <code>-1</code>.
	 */
	public FileMetaDataImpl(FileMetaData parent, String name, FileOwner owner, FileMode mode, long length,
			Date lastModifiedDate) {
		
		super();

		if (parent == null)
			throw new IllegalArgumentException("Argument path is null.");
		if (!parent.isDirectory())
			throw new IllegalArgumentException("Argument parent |" + parent.getAbsolutePath() + "| is not a directory.");
		if (name == null)
			throw new IllegalArgumentException("Argument name is null.");
		if (name.isEmpty())
			throw new IllegalArgumentException("Argument name is empty.");
		if (name.indexOf('/') != -1)
			throw new IllegalArgumentException("Argument name contains a slash: " + name);
		if (owner == null)
			throw new IllegalArgumentException("Argument owner is null.");
		if (mode == null)
			throw new IllegalArgumentException("Argument mode is null.");
		if (length < -1)
			throw new IllegalArgumentException("Argument length is less than -1: " + length);
		if (lastModifiedDate == null)
			throw new IllegalArgumentException("Argument lastModifiedDate is null.");
		
		this.path = parent.getAbsolutePath();
		this.name = name;
		this.directory = false;
		this.targetPath = null;
		this.owner = owner;
		this.mode = mode;
		this.length = length;
		this.lastModifiedDate = lastModifiedDate;
	}


	/**
	 * <p>
	 * Creates the meta data of a symbolic link.
	 * </p>
	 *
	 * @param path
	 *            The path of the file excluding the name.
	 * @param name
	 *            The name of the file.
	 * @param targetPath
	 *            The target path of the symbolic link.
	 * @param owner
	 *            The owner of the file.
	 * @param mode
	 *            The mode of the file.
	 * @param lastModifiedDate
	 *            The timestamp of the last modification.
	 * @throws IllegalArgumentException
	 *             If any of the parameters are <code>null</code>, the path is
	 *             empty, the name contains a slash, the path is empty or if the
	 *             target path is empty.
	 */
	public FileMetaDataImpl(String path, String name, String targetPath, FileOwner owner, FileMode mode,
			Date lastModifiedDate) {
		
		super();

		if (path == null)
			throw new IllegalArgumentException("Argument path is null.");
		if (path.isEmpty())
			throw new IllegalArgumentException("Argument path is empty.");
		if (name == null)
			throw new IllegalArgumentException("Argument name is null.");
		if (name.indexOf('/') != -1)
			throw new IllegalArgumentException("Argument name contains a slash: " + name);
		if (targetPath == null)
			throw new IllegalArgumentException("Argument targetPath is null.");
		if (targetPath.isEmpty())
			throw new IllegalArgumentException("Argument targetPath is empty.");
		if (owner == null)
			throw new IllegalArgumentException("Argument owner is null.");
		if (mode == null)
			throw new IllegalArgumentException("Argument mode is null.");
		if (lastModifiedDate == null)
			throw new IllegalArgumentException("Argument lastModifiedDate is null.");
		
		this.path = this.revisePath(path);
		this.name = name;
		this.directory = false;
		this.targetPath = targetPath;
		this.owner = owner;
		this.mode = mode;
		this.length = -1;
		this.lastModifiedDate = lastModifiedDate;
	}
	
	
	/**
	 * <p>
	 * Creates the meta data of a symbolic link.
	 * </p>
	 *
	 * @param parent
	 *            The parent directory.
	 * @param name
	 *            The name of the file.
	 * @param targetPath
	 *            The target path of the symbolic link.
	 * @param owner
	 *            The owner of the file.
	 * @param mode
	 *            The mode of the file.
	 * @param lastModifiedDate
	 *            The timestamp of the last modification.
	 * @throws IllegalArgumentException
	 *             If any of the parameters are <code>null</code>, the parent is
	 *             not a directory, the name contains a slash, the name is empty
	 *             or the target path is empty.
	 */
	public FileMetaDataImpl(FileMetaData parent, String name, String targetPath, FileOwner owner, FileMode mode,
			Date lastModifiedDate) {
		super();

		if (parent == null)
			throw new IllegalArgumentException("Argument path is null.");
		if (!parent.isDirectory())
			throw new IllegalArgumentException("Argument parent |" + parent.getAbsolutePath() + "| is not a directory.");
		if (name == null)
			throw new IllegalArgumentException("Argument name is null.");
		if (name.isEmpty())
			throw new IllegalArgumentException("An empty name is only valid for the root. Path: " + parent.getAbsolutePath());
		if (name.indexOf('/') != -1)
			throw new IllegalArgumentException("Argument name contains a slash: " + name);
		if (targetPath == null)
			throw new IllegalArgumentException("Argument targetPath is null.");
		if (targetPath.isEmpty())
			throw new IllegalArgumentException("Argument targetPath is empty.");
		if (owner == null)
			throw new IllegalArgumentException("Argument owner is null.");
		if (mode == null)
			throw new IllegalArgumentException("Argument mode is null.");
		if (lastModifiedDate == null)
			throw new IllegalArgumentException("Argument lastModifiedDate is null.");
		
		this.path = parent.getAbsolutePath();
		this.name = name;
		this.directory = false;
		this.targetPath = targetPath;
		this.owner = owner;
		this.mode = mode;
		this.length = -1;
		this.lastModifiedDate = lastModifiedDate;
	}


	/**
	 * <p>
	 * Revises the path by making sure that it starts with a slash and ends with
	 * a slash.
	 * </p>
	 *
	 * @param path
	 *            The path.
	 * @return The revised path.
	 */
	private String revisePath(String path) {
		String result;
		
		
		result = path;
		if (result.startsWith("./")) {
			result = result.substring(1);
		}
		if (!result.startsWith("/")) {
			result = '/' + result;
		}
		if (!result.endsWith("/")) {
			result = result + '/';
		}
		
		return result;
	}


	@Override
	public String getPath() {
		return this.path;
	}
	
	
	@Override
	public String getName() {
		return this.name;
	}


	@Override
	public String getAbsolutePath() {
		StringBuilder sb;
		
		
		sb = new StringBuilder();
		sb.append(this.path);
		sb.append(this.name);
		if (this.directory && !this.name.isEmpty()) {
			sb.append('/');
		}
		return sb.toString();
	}


	@Override
	public boolean isFile() {
		return !this.directory;
	}


	@Override
	public boolean isDirectory() {
		return this.directory;
	}


	@Override
	public boolean isSymbolicLink() {
		return (this.targetPath != null);
	}


	@Override
	public String getTargetPath() {
		return this.targetPath;
	}
	
	
	@Override
	public FileOwner getOwner() {
		return this.owner;
	}


	@Override
	public FileMode getMode() {
		return this.mode;
	}


	@Override
	public String getModeAsText() {
		StringBuilder sb;


		sb = new StringBuilder();
		if (this.directory) {
			sb.append('d');
		} else if (this.targetPath != null) {
			sb.append('l');
		} else {
			sb.append('-');
		}
		sb.append(this.mode.getText());

		return sb.toString();
	}
	
	
	@Override
	public long getLength() {
		return this.length;
	}


	@Override
	public Date getLastModifiedDate() {
		return this.lastModifiedDate;
	}


	/**
	 * <p>
	 * Creates the meta data for a directory.
	 * </p>
	 *
	 * @param absolutePath
	 *            The absolute path.
	 * @param owner
	 *            The owner of the file.
	 * @param mode
	 *            The mode of the file.
	 * @param lastModifiedDate
	 *            The timestamp of the last modification.
	 * @return The meta data.
	 * @throws IllegalArgumentException
	 *             If any of the parameters are <code>null</code> or the
	 *             absolute path is empty.
	 */
	public static FileMetaData createDirectoryMetaData(String absolutePath, FileOwner owner, FileMode mode,
			Date lastModifiedDate) {
		
		FileMetaData directory;
		String[] paths;
		
		
		if (absolutePath == null)
			throw new IllegalArgumentException("Argument absolutePath is null.");
		if (absolutePath.isEmpty())
			throw new IllegalArgumentException("Argument absolutePath is empty.");
		if (owner == null)
			throw new IllegalArgumentException("Argument owner is null.");
		if (mode == null)
			throw new IllegalArgumentException("Argument mode is null.");
		if (lastModifiedDate == null)
			throw new IllegalArgumentException("Argument lastModifiedDate is null.");
		
		paths = splitAbsolutePath(absolutePath);
		if (paths[1].endsWith("/")) {
			paths[1] = paths[1].substring(0, paths[1].length() - 1);
		}
		directory = new FileMetaDataImpl(paths[0], paths[1], owner, mode, lastModifiedDate);
		return directory;
	}


	/**
	 * <p>
	 * Creates the meta data for a regular file.
	 * </p>
	 *
	 * @param absolutePath
	 *            The absolute path.
	 * @param owner
	 *            The owner of the file.
	 * @param mode
	 *            The mode of the file.
	 * @param length
	 *            The length of the file or <code>-1</code>, if the length is
	 *            unknown.
	 * @param lastModifiedDate
	 *            The timestamp of the last modification.
	 * @return The meta data.
	 * @throws IllegalArgumentException
	 *             If any of the parameters are <code>null</code>, the absolute
	 *             path is empty or the length is less than <code>-1</code>.
	 */
	public static FileMetaData createFileMetaData(String absolutePath, FileOwner owner, FileMode mode, long length,
			Date lastModifiedDate) {
		
		FileMetaData file;
		String[] paths;


		if (absolutePath == null)
			throw new IllegalArgumentException("Argument absolutePath is null.");
		if (absolutePath.isEmpty())
			throw new IllegalArgumentException("Argument absolutePath is empty.");
		if (owner == null)
			throw new IllegalArgumentException("Argument owner is null.");
		if (mode == null)
			throw new IllegalArgumentException("Argument mode is null.");
		if (length < -1)
			throw new IllegalArgumentException("Argument length is less than -1: " + length);
		if (lastModifiedDate == null)
			throw new IllegalArgumentException("Argument lastModifiedDate is null.");
		
		paths = splitAbsolutePath(absolutePath);
		file = new FileMetaDataImpl(paths[0], paths[1], owner, mode, length, lastModifiedDate);
		return file;
	}
	
	
	/**
	 * <p>
	 * Creates the meta data for a symbolic link.
	 * </p>
	 *
	 * @param absolutePath
	 *            The absolute path.
	 * @param targetPath
	 *            The target path of the symbolic link.
	 * @param owner
	 *            The owner of the symbolic link.
	 * @param mode
	 *            The mode of the symbolic link.
	 * @param lastModifiedDate
	 *            The timestamp of the last modification.
	 * @return The meta data.
	 * @throws IllegalArgumentException
	 *             If any of the parameters are <code>null</code>, the absolute
	 *             path is empty or the target is empty.
	 */
	public static FileMetaData createSymbolicLinkMetaData(String absolutePath, String targetPath, FileOwner owner,
			FileMode mode, Date lastModifiedDate) {
		
		FileMetaData symbolicLink;
		String[] paths;


		if (absolutePath == null)
			throw new IllegalArgumentException("Argument absolutePath is null.");
		if (absolutePath.isEmpty())
			throw new IllegalArgumentException("Argument absolutePath is empty.");
		if (targetPath == null)
			throw new IllegalArgumentException("Argument targetPath is null.");
		if (targetPath.isEmpty())
			throw new IllegalArgumentException("Argument targetPath is empty.");
		if (owner == null)
			throw new IllegalArgumentException("Argument owner is null.");
		if (mode == null)
			throw new IllegalArgumentException("Argument mode is null.");
		if (lastModifiedDate == null)
			throw new IllegalArgumentException("Argument lastModifiedDate is null.");
		
		paths = splitAbsolutePath(absolutePath);
		symbolicLink = new FileMetaDataImpl(paths[0], paths[1], targetPath, owner, mode, lastModifiedDate);
		return symbolicLink;
	}
	
	
	/**
	 * <p>
	 * Splits the absolute path into the path and the name of the file.
	 * </p>
	 *
	 * @param absolutePath
	 *            The absolute path.
	 * @return An array with the path as first element and the name as second
	 *         element.
	 */
	private static String[] splitAbsolutePath(String absolutePath) {
		String[] result;
		String value;
		int index;


		result = new String[2];
		
		// --- Make sure the path starts with a slash ---
		value = absolutePath;
		if (value.startsWith("./")) {
			value = value.substring(1);
		}
		if (!value.startsWith("/")) {
			value = "/" + value;
		}
		// --- Do we have the root? ---
		if (value.length() == 1) {
			result[0] = "/";
			result[1] = "";
		}
		// --- Otherwise split the path ---
		else {
			index = value.lastIndexOf('/');
			// --- Do we have a directory path? ---
			if (index == (value.length() - 1)) {
				index = value.lastIndexOf('/', index - 1);
			}
			result[0] = value.substring(0, index + 1);
			result[1] = value.substring(index + 1);
		}
		return result;
	}


}
