/*
 * dpkg - Debian Package library and the Debian Package Maven plugin
 * (c) Copyright 2016 Gerrit Hohl
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package net.sourceforge.javadpkg.io.impl;

import java.io.File;
import java.io.IOException;

import net.sourceforge.javadpkg.io.DataSource;
import net.sourceforge.javadpkg.io.DataSwap;
import net.sourceforge.javadpkg.io.DataTarget;


/**
 * <p>
 * A {@link DataSwap} implementation which uses temporary files for storing the
 * data.
 * </p>
 *
 * @see File#createTempFile(String, String)
 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
 * @version <b>1.0</b>, 29.04.2016 by Gerrit Hohl
 */
public class DataTempFileSwap implements DataSwap {
	
	
	/** The name. */
	private String		name;
	/** The prefix. */
	private String		prefix;
	/** The suffix. */
	private String		suffix;
	
	/** The file. */
	private File		file;
	/** The target. */
	private DataTarget	target;
	/** The source. */
	private DataSource	source;
	/** The flag if the swap was closed. */
	private boolean		closed;
	
	
	/**
	 * <p>
	 * Creates a swap.
	 * </p>
	 *
	 * @param name
	 *            The name of the target and source.
	 * @throws IllegalArgumentException
	 *             If the name is <code>null</code>.
	 */
	public DataTempFileSwap(String name) {
		super();
		
		if (name == null)
			throw new IllegalArgumentException("Argument name is null.");
		
		this.name = name;
		this.prefix = "data-temp-file-swap-";
		this.suffix = ".tmp";

		this.file = null;
		this.target = null;
		this.source = null;
		this.closed = false;
	}
	
	
	/**
	 * <p>
	 * Creates a swap.
	 * </p>
	 *
	 * @param name
	 *            The name of the target and source.
	 * @param prefix
	 *            The prefix of the filename.
	 * @param suffix
	 *            The suffix of the filename.
	 * @throws IllegalArgumentException
	 *             If any of the parameters are <code>null</code>.
	 */
	public DataTempFileSwap(String name, String prefix, String suffix) {
		this(name);
		
		if (prefix == null)
			throw new IllegalArgumentException("Argument prefix is null.");
		if (suffix == null)
			throw new IllegalArgumentException("Argument suffix is null.");
		
		this.prefix = prefix;
		this.suffix = suffix;
	}
	
	
	@Override
	public DataTarget getTarget() throws IOException {
		if (this.closed)
			throw new IllegalStateException("Can't return the target because the swap |" + this.name + "| was already closed.");
		if (this.source != null)
			throw new IllegalStateException("Can't return the target of the swap |" + this.name
					+ "| because the corresponding source is already open.");
		if (this.target != null)
			return this.target;
		
		this.file = File.createTempFile(this.prefix, this.suffix);
		this.target = new DataFileTarget(this.file);
		return this.target;
	}


	@Override
	public DataSource getSource() throws IOException {
		if (this.closed)
			throw new IllegalStateException("Can't return the source because the swap |" + this.name + "| was already closed.");
		if (this.source != null)
			return this.source;
		if (this.target == null)
			throw new IllegalStateException("Can't return the source of the swap |" + this.name
					+ "| because the corresponding target hasn't been opened yet.");
		
		this.target.close();
		this.source = new DataFileSource(this.file, true);
		return this.source;
	}
	
	
	@Override
	public void close() throws IOException {
		try {
			if (this.target != null) {
				try {
					this.target.close();
				} catch (IOException e) {
					throw new IOException("Couldn't close target of swap |" + this.name + "|: " + e.getMessage(), e);
				}
			}
			if (this.source != null) {
				try {
					this.source.close();
				} catch (IOException e) {
					throw new IOException("Couldn't close source of swap |" + this.name + "|: " + e.getMessage(), e);
				}
			}
			if ((this.file != null) && this.file.exists()) {
				if (!this.file.delete())
					throw new IOException(
							"Couldn't delete swap |" + this.name + "| (Path: |" + this.file.getAbsolutePath() + "|).");
			}
		} finally {
			this.file = null;
			this.target = null;
			this.source = null;
			this.closed = true;
		}
	}


}
