/*
 * dpkg - Debian Package library and the Debian Package Maven plugin
 * (c) Copyright 2015 Gerrit Hohl
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package net.sourceforge.javadpkg.io.impl;

import java.io.IOException;
import java.io.InputStream;

import net.sourceforge.javadpkg.io.DataSource;

/**
 * <p>
 * A {@link DataSource} implementation based on a stream.
 * </p>
 *
 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
 * @version <b>1.0</b>, 31.12.2015 by Gerrit Hohl
 */
public class DataStreamSource implements DataSource {
	
	
	/** The stream. */
	private InputStream				in;
	/**
	 * <p>
	 * The public stream which is returned by the {@link #getInputStream()}
	 * method.
	 * </p>
	 */
	private UncloseableInputStream	publicIn;
	/** The name of the source. */
	private String					name;
	/** The flag if the {@link #close()} method will close the stream. */
	private boolean					closeable;
									
									
	/**
	 * <p>
	 * Creates a source.
	 * </p>
	 *
	 * @param in
	 *            The stream.
	 * @param name
	 *            The name of the source.
	 * @param closeable
	 *            The flag if the {@link #close()} method will close the stream.
	 * @throws IllegalArgumentException
	 *             If any of the parameters are <code>null</code>.
	 */
	public DataStreamSource(InputStream in, String name, boolean closeable) {
		super();
		
		if (in == null)
			throw new IllegalArgumentException("Argument in is null.");
		if (name == null)
			throw new IllegalArgumentException("Argument name is null.");

		this.in = in;
		this.name = name;
		this.closeable = closeable;

		this.createPublicInputStream();
	}


	@Override
	public String getName() {
		return this.name;
	}


	@Override
	public long getLength() {
		return -1;
	}
	
	
	@Override
	public boolean isResettable() {
		return false;
	}
	
	
	@Override
	public void reset() throws IOException {
		throw new IOException("Source |" + this.name + "| doesn't support a reset.");
	}
	
	
	/**
	 * <p>
	 * Creates the {@link InputStream} which is returned by the
	 * {@link #getInputStream()} method.
	 * </p>
	 */
	private void createPublicInputStream() {
		this.publicIn = new UncloseableInputStream(this.in, new DelegateCloseHandler(this));
	}
	
	
	@Override
	public InputStream getInputStream() throws IOException {
		return this.publicIn;
	}


	@Override
	public void close() throws IOException {
		try {
			if ((this.in != null) && this.closeable) {
				this.in.close();
			}
		} finally {
			this.publicIn = null;
			this.in = null;
		}
	}


}
