/*
 * dpkg - Debian Package library and the Debian Package Maven plugin
 * (c) Copyright 2016 Gerrit Hohl
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package net.sourceforge.javadpkg.io.impl;

import java.io.IOException;
import java.io.OutputStream;

import net.sourceforge.javadpkg.io.DataConsumer;


/**
 * <p>
 * A {@link DataConsumer} which writes the data into an {@link OutputStream}.
 * </p>
 *
 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
 * @version <b>1.0</b>, 27.04.2016 by Gerrit Hohl
 */
public class DataStreamConsumer implements DataConsumer {
	
	
	/** The output stream. */
	private OutputStream	out;
	/** The name of the consumer in the exception (if one is thrown). */
	private String			name;


	/**
	 * <p>
	 * Creates a consumer.
	 * </p>
	 *
	 * @param out
	 *            The output stream.
	 * @param name
	 *            The name of the handler in the exception (if one is thrown).
	 * @throws IllegalArgumentException
	 *             If any of the parameters are <code>null</code>.
	 */
	public DataStreamConsumer(OutputStream out, String name) {
		super();

		if (out == null)
			throw new IllegalArgumentException("Argument out is null.");
		if (name == null)
			throw new IllegalArgumentException("Argument name is null.");
			
		this.out = out;
		this.name = name;
	}


	@Override
	public String getName() {
		return this.name;
	}


	@Override
	public void consume(byte[] buffer, int length) throws IOException {
		if (buffer == null)
			throw new IllegalArgumentException("Argument buffer is null.");
		if (length < 0)
			throw new IllegalArgumentException("Argument length is negative: " + length);
		if (length > buffer.length)
			throw new IllegalArgumentException("Argument length is greater than the buffer length. Length: " + length
					+ "; buffer length: " + buffer.length);

		try {
			this.out.write(buffer, 0, length);
		} catch (IOException e) {
			throw new IOException("Couldn't write " + length + " byte(s) to " + this.name + ": " + e.getMessage(), e);
		}
	}
	
	
}
