/*
 * dpkg - Debian Package library and the Debian Package Maven plugin
 * (c) Copyright 2016 Gerrit Hohl
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package net.sourceforge.javadpkg.io.impl;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;

import net.sourceforge.javadpkg.io.DataSource;
import net.sourceforge.javadpkg.io.DataSwap;
import net.sourceforge.javadpkg.io.DataTarget;


/**
 * <p>
 * A {@link DataSwap} implementation which uses a byte array for storing the
 * data.
 * </p>
 *
 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
 * @version <b>1.0</b>, 02.05.2016 by Gerrit Hohl
 */
public class DataByteArraySwap implements DataSwap {
	
	
	/** The name. */
	private String					name;

	/** The stream. */
	private ByteArrayOutputStream	out;
	/** The target. */
	private DataTarget				target;
	/** The source. */
	private DataSource				source;
	/** The flag if the swap was closed. */
	private boolean					closed;
	
	
	/**
	 * <p>
	 * Creates a swap.
	 * </p>
	 *
	 * @param name
	 *            The name of the target and source.
	 * @throws IllegalArgumentException
	 *             If the name is <code>null</code>.
	 */
	public DataByteArraySwap(String name) {
		super();

		if (name == null)
			throw new IllegalArgumentException("Argument name is null.");
		
		this.name = name;
	}
	
	
	@Override
	public DataTarget getTarget() throws IOException {
		if (this.closed)
			throw new IllegalStateException("Can't return the target because the swap |" + this.name + "| was already closed.");
		if (this.source != null)
			throw new IllegalStateException("Can't return the target of the swap |" + this.name
					+ "| because the corresponding source is already open.");
		if (this.target != null)
			return this.target;
		
		this.out = new ByteArrayOutputStream();
		this.target = new DataStreamTarget(this.out, this.name, true);
		return this.target;
	}


	@Override
	public DataSource getSource() throws IOException {
		if (this.closed)
			throw new IllegalStateException("Can't return the source because the swap |" + this.name + "| was already closed.");
		if (this.source != null)
			return this.source;
		if (this.target == null)
			throw new IllegalStateException("Can't return the source of the swap |" + this.name
					+ "| because the corresponding target hasn't been opened yet.");
		
		this.target.close();
		this.source = new DataStreamSource(new ByteArrayInputStream(this.out.toByteArray()), this.name, true);
		return this.source;
	}
	
	
	@Override
	public void close() throws IOException {
		try {
			if (this.target != null) {
				try {
					this.target.close();
				} catch (IOException e) {
					throw new IOException("Couldn't close target of swap |" + this.name + "|: " + e.getMessage(), e);
				}
			}
			if (this.source != null) {
				try {
					this.source.close();
				} catch (IOException e) {
					throw new IOException("Couldn't close source of swap |" + this.name + "|: " + e.getMessage(), e);
				}
			}
		} finally {
			this.out = null;
			this.target = null;
			this.source = null;
			this.closed = true;
		}
	}


}
