/*
 * dpkg - Debian Package library and the Debian Package Maven plugin
 * (c) Copyright 2016 Gerrit Hohl
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package net.sourceforge.javadpkg.io.impl;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;

import net.sourceforge.javadpkg.io.DataSource;


/**
 * <p>
 * A {@link DataSource} implementation based on an array.
 * </p>
 *
 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
 * @version <b>1.0</b>, 26.04.2016 by Gerrit Hohl
 */
public class DataByteArraySource implements DataSource {
	
	
	/** The array. */
	private byte[]		data;
	/** The stream. */
	private InputStream	in;
	/** The name of the source. */
	private String		name;
	/** The flag if the {@link #close()} method will close the stream. */
	private boolean		closeable;
	/** The flag if the {@link #reset()} method will be supported. */
	private boolean		resettable;
	
	
	/**
	 * <p>
	 * Creates a source.
	 * </p>
	 *
	 * @param data
	 *            The array.
	 * @param name
	 *            The name of the source.
	 * @param closeable
	 *            The flag if the {@link #close()} method will close the stream.
	 * @param resettable
	 *            The flag if the {@link #reset()} method will be supported.
	 * @throws IllegalArgumentException
	 *             If any of the parameters are <code>null</code>.
	 */
	public DataByteArraySource(byte[] data, String name, boolean closeable, boolean resettable) {
		super();
		
		if (data == null)
			throw new IllegalArgumentException("Argument data is null.");
		if (name == null)
			throw new IllegalArgumentException("Argument name is null.");
		
		this.data = data.clone();
		this.in = new ByteArrayInputStream(this.data);
		this.name = name;
		this.closeable = closeable;
		this.resettable = resettable;
	}
	
	
	@Override
	public String getName() {
		return this.name;
	}
	
	
	@Override
	public long getLength() {
		return this.data.length;
	}
	
	
	@Override
	public boolean isResettable() {
		return this.resettable;
	}
	
	
	@Override
	public void reset() throws IOException {
		if (!this.resettable)
			throw new IOException("Source |" + this.name + "| doesn't support a reset.");
		if (this.in != null) {
			try {
				this.in.close();
			} finally {
				this.in = null;
			}
		}
		this.in = new ByteArrayInputStream(this.data);
	}
	
	
	@Override
	public InputStream getInputStream() throws IOException {
		return this.in;
	}
	
	
	@Override
	public void close() throws IOException {
		try {
			if ((this.in != null) && this.closeable) {
				this.in.close();
			}
		} finally {
			if (this.closeable) {
				this.in = null;
			}
		}
	}


}
