/*
 * dpkg - Debian Package library and the Debian Package Maven plugin
 * (c) Copyright 2016 Gerrit Hohl
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package net.sourceforge.javadpkg.io;


/**
 * <p>
 * The mode of a file.
 * </p>
 *
 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
 * @version <b>1.0</b>, 12.05.2016 by Gerrit Hohl
 * @version <b>1.0</b>, 21.06.2018 by Gerrit Hohl
 */
public interface FileMode {
	
	
	/**
	 * <p>
	 * Returns the mode as number including all octals.
	 * </p>
	 *
	 * @return The mode.
	 */
	int getMode();


	/**
	 * <p>
	 * Returns the mode as an octal number consisting only of the last 3 octals.
	 * </p>
	 * <p>
	 * The returned value may look like this: <blockquote>
	 * <code>777</code></blockquote> or like this: <blockquote>
	 * <code>000</code></blockquote>
	 * </p>
	 *
	 * @return The mode.
	 */
	String getOctal();


	/**
	 * <p>
	 * Returns the mode as textual representation excluding the &quot;sticky
	 * bit&quot;.
	 * </p>
	 * <p>
	 * The returned value may look like this: <blockquote>
	 * <code>rwxrwxrwx</code></blockquote> or like this: <blockquote>
	 * <code>---------</code></blockquote>
	 * </p>
	 *
	 * @return The mode.
	 */
	String getText();
	
	
	/**
	 * <p>
	 * Returns the sticky bit.
	 * </p>
	 * <p>
	 * See <a href=
	 * "https://en.wikipedia.org/wiki/File_system_permissions#Changing_permission_behavior_with_setuid.2C_setgid.2C_and_sticky_bits">
	 * Changing permission behavior with setuid, setgid, and sticky bits</a> for
	 * more information.
	 * </p>
	 *
	 * @return The sticky bit: A number from octal 0 (decimal: 0) to octal 7
	 *         (decimal: 7).
	 */
	int getStickyBit();


	/**
	 * <p>
	 * Returns the flag if the owner can read the file.
	 * </p>
	 *
	 * @return The flag: <code>true</code>, if the owner can, <code>false</code>
	 *         otherwise.
	 */
	boolean isOwnerReadable();


	/**
	 * <p>
	 * Returns the flag if the owner can write the file.
	 * </p>
	 *
	 * @return The flag: <code>true</code>, if the owner can, <code>false</code>
	 *         otherwise.
	 */
	boolean isOwnerWriteable();


	/**
	 * <p>
	 * Returns the flag if the owner can execute the file.
	 * </p>
	 *
	 * @return The flag: <code>true</code>, if the owner can, <code>false</code>
	 *         otherwise.
	 */
	boolean isOwnerExecutable();


	/**
	 * <p>
	 * Returns the flag if the group can read the file.
	 * </p>
	 *
	 * @return The flag: <code>true</code>, if the group can, <code>false</code>
	 *         otherwise.
	 */
	boolean isGroupReadable();


	/**
	 * <p>
	 * Returns the flag if the group can write the file.
	 * </p>
	 *
	 * @return The flag: <code>true</code>, if the group can, <code>false</code>
	 *         otherwise.
	 */
	boolean isGroupWriteable();


	/**
	 * <p>
	 * Returns the flag if the group can execute the file.
	 * </p>
	 *
	 * @return The flag: <code>true</code>, if the group can, <code>false</code>
	 *         otherwise.
	 */
	boolean isGroupExecutable();


	/**
	 * <p>
	 * Returns the flag if the others can read the file.
	 * </p>
	 *
	 * @return The flag: <code>true</code>, if the others can,
	 *         <code>false</code> otherwise.
	 */
	boolean isOtherReadable();


	/**
	 * <p>
	 * Returns the flag if the others can write the file.
	 * </p>
	 *
	 * @return The flag: <code>true</code>, if the others can,
	 *         <code>false</code> otherwise.
	 */
	boolean isOtherWriteable();


	/**
	 * <p>
	 * Returns the flag if the others can execute the file.
	 * </p>
	 *
	 * @return The flag: <code>true</code>, if the others can,
	 *         <code>false</code> otherwise.
	 */
	boolean isOtherExecutable();


}
