/*
 * dpkg - Debian Package library and the Debian Package Maven plugin
 * (c) Copyright 2016 Gerrit Hohl
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package net.sourceforge.javadpkg.io;

import java.util.Date;

/**
 * <p>
 * The meta data of a file or directory.
 * </p>
 *
 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
 * @version <b>1.0</b>, 12.05.2016 by Gerrit Hohl
 */
public interface FileMetaData {
	
	
	/**
	 * <p>
	 * Returns the path of the file excluding the name.
	 * </p>
	 *
	 * @return The path.
	 * @see #getName()
	 * @see #getAbsolutePath()
	 */
	String getPath();
	
	
	/**
	 * <p>
	 * Returns the name of the file.
	 * </p>
	 *
	 * @return The name.
	 * @see #getPath()
	 * @see #getAbsolutePath()
	 */
	String getName();
	
	
	/**
	 * <p>
	 * Returns the absolute path consisting of the path and the name of the
	 * file.
	 * </p>
	 * <p>
	 * If the file is a directory a trailing slash will be added.
	 * </p>
	 *
	 * @return The absolute path.
	 * @see #getPath()
	 * @see #getName()
	 */
	String getAbsolutePath();


	/**
	 * <p>
	 * Returns the flag if the file is a regular file.
	 * </p>
	 * <p>
	 * A symbolic link is also a file.
	 * </p>
	 *
	 * @return The flag: <code>true</code>, if the file is a regular file,
	 *         <code>false</code> otherwise.
	 * @see #isDirectory()
	 * @see #isSymbolicLink()
	 */
	boolean isFile();


	/**
	 * <p>
	 * Returns the flag if the file is a directory.
	 * </p>
	 *
	 * @return The flag: <code>true</code>, if the file is a directory,
	 *         <code>false</code> otherwise.
	 * @see #isFile()
	 */
	boolean isDirectory();


	/**
	 * <p>
	 * Returns the flag if the file is a symbolic link.
	 * </p>
	 * 
	 * @return The flag: <code>true</code>, if the file is a symbolic link,
	 *         <code>false</code> otherwise.
	 * @see #isFile()
	 * @see #getTargetPath()
	 */
	boolean isSymbolicLink();


	/**
	 * <p>
	 * Returns the target path of the symbolic link.
	 * </p>
	 * 
	 * @return The target path or <code>null</code>, if this meta data does not
	 *         represent a symbolic link.
	 * @see #isSymbolicLink()
	 */
	String getTargetPath();
	
	
	/**
	 * <p>
	 * Returns the owner of the file.
	 * </p>
	 *
	 * @return The owner.
	 */
	FileOwner getOwner();


	/**
	 * <p>
	 * Returns the mode of the file.
	 * </p>
	 *
	 * @return The mode.
	 */
	FileMode getMode();


	/**
	 * <p>
	 * Returns the textual representation of the mode including the directory
	 * flag.
	 * </p>
	 * <p>
	 * The returned value may look like this: <blockquote>
	 * <code>drwxrwxrwx</code></blockquote> or like this: <blockquote>
	 * <code>----------</code></blockquote>
	 * </p>
	 *
	 * @return The mode.
	 * @see FileMode#getText()
	 */
	String getModeAsText();
	
	
	/**
	 * <p>
	 * Returns the of the file in bytes.
	 * </p>
	 *
	 * @return The length or <code>-1</code>, if the length is unknown or the
	 *         file is not a regular file.
	 */
	long getLength();
	
	
	/**
	 * <p>
	 * Returns the timestamp of the last modification.
	 * </p>
	 *
	 * @return The timestamp.
	 */
	Date getLastModifiedDate();
	
	
}
