/*
 * dpkg - Debian Package library and the Debian Package Maven plugin
 * (c) Copyright 2015 Gerrit Hohl
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package net.sourceforge.javadpkg.io;

import java.io.Closeable;
import java.io.IOException;
import java.io.InputStream;

/**
 * <p>
 * The source for reading data.
 * </p>
 *
 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
 * @version <b>1.0</b>, 31.12.2015 by Gerrit Hohl
 */
public interface DataSource extends Closeable {


	/**
	 * <p>
	 * Returns the name of the source.
	 * </p>
	 * <p>
	 * This name is used e.g. for detail messages of exceptions.
	 * </p>
	 *
	 * @return The name.
	 */
	String getName();


	/**
	 * <p>
	 * Returns the length of the data which can be read from the source in
	 * bytes.
	 * </p>
	 *
	 * @return The length or <code>-1</code>, if the length is unknown.
	 */
	long getLength();


	/**
	 * <p>
	 * Returns the flag if the source can be reseted.
	 * </p>
	 *
	 * @return The flag: <code>true</code>, if the source can be reseted,
	 *         <code>false</code> otherwise.
	 * @see #reset()
	 */
	boolean isResettable();


	/**
	 * <p>
	 * Resets the source.
	 * </p>
	 * <p>
	 * If the source can't be reseted an {@link IOException} will be thrown.
	 * </p>
	 * <p>
	 * If the source can be reseted the next call of the
	 * {@link #getInputStream()} method will return a new {@link InputStream}
	 * which reads the source from the beginning again.
	 * </p>
	 * 
	 * @throws IOException
	 *             If an I/O error occurs or if the source doesn't support a
	 *             reset.
	 * @see #isResettable()
	 */
	void reset() throws IOException;


	/**
	 * <p>
	 * Returns a stream for reading the data.
	 * </p>
	 * <p>
	 * The method will always return the same {@link InputStream} if the source
	 * is still open. If the {@link InputStream#close()} method is called the
	 * source will behave the same as the {@link #close()} method was called.
	 * </p>
	 * <p>
	 * If the {@link #reset()} method was successfully called the method will
	 * return a different {@link InputStream}.
	 * </p>
	 *
	 * @return The stream.
	 * @throws IOException
	 *             If an I/O error occurs.
	 */
	InputStream getInputStream() throws IOException;


}
