/*
 * dpkg - Debian Package library and the Debian Package Maven plugin
 * (c) Copyright 2016 Gerrit Hohl
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package net.sourceforge.javadpkg.impl;

import java.util.Map;
import java.util.TreeMap;

import net.sourceforge.javadpkg.Context;
import net.sourceforge.javadpkg.ParseException;
import net.sourceforge.javadpkg.TemplateType;
import net.sourceforge.javadpkg.TemplateTypeParser;


/**
 * <p>
 * A {@link TemplateTypeParser} implementation.
 * </p>
 *
 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
 * @version <b>1.0</b>, 29.04.2016 by Gerrit Hohl
 */
public class TemplateTypeParserImpl implements TemplateTypeParser {


	/** The key for thy type &quot;select&quot;. */
	private static final String			KEY_SELECT		= "select";
	/** The key for the type &quot;multiselect&quot;. */
	private static final String			KEY_MULTISELECT	= "multiselect";
														
														
	/** The supported types. */
	private Map<String, TemplateType>	types;
										
										
	/**
	 * <p>
	 * Creates a parser.
	 * </p>
	 */
	public TemplateTypeParserImpl() {
		super();

		this.types = new TreeMap<>();
		// --- Any arbitrary string of data. ---
		this.addType("string");
		// --- A finite number of possible values. ---
		this.addType(KEY_SELECT);
		// --- A finite number of possible values from which the user can select any number of items. ---
		this.addType(KEY_MULTISELECT);
		// --- A boolean value. ---
		this.addType("boolean");
		// --- A note which should be shown to the user. ---
		this.addType("note");
		// --- A text which can be shown to the user. ---
		this.addType("text");
		// --- A password. ---
		this.addType("password");
		// --- An error. ---
		this.addType("error");
		// --- A title. ---
		this.addType("title");
	}


	/**
	 * <p>
	 * Adds a type.
	 * </p>
	 *
	 * @param text
	 *            The text.
	 */
	private void addType(String text) {
		this.types.put(text, new TemplateTypeImpl(text));
	}


	@Override
	public TemplateType parseTemplateType(String value, Context context) throws ParseException {
		TemplateType type;
		
		
		if (value == null)
			throw new IllegalArgumentException("Argument value is null.");
		if (context == null)
			throw new IllegalArgumentException("Argument context is null.");

		type = this.types.get(value);
		if (type == null) {
			context.addWarning(new TemplateTypeUnsupportedWarning(value));
			type = new TemplateTypeImpl(value);
		}
		
		return type;
	}
	
	
	@Override
	public TemplateType getTypeSelect() {
		return this.types.get(KEY_SELECT);
	}


	@Override
	public TemplateType getTypeMultiselect() {
		return this.types.get(KEY_MULTISELECT);
	}


	/* **********************************************************************
	 * **********************************************************************
	 * **********************************************************************
	 * **********************************************************************
	 * **********************************************************************
	 */


	/**
	 * <p>
	 * The {@link TemplateType} implementation of this class.
	 * </p>
	 *
	 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
	 * @version <b>1.0</b>, 29.04.2016 by Gerrit Hohl
	 */
	private class TemplateTypeImpl implements TemplateType {


		/** The text. */
		private String text;


		/**
		 * <p>
		 * Creates a template type.
		 * </p>
		 *
		 * @param text
		 *            The text.
		 */
		public TemplateTypeImpl(String text) {
			super();

			this.text = text;
		}


		@Override
		public String getText() {
			return this.text;
		}
		
		
	}


}
