/*
 * dpkg - Debian Package library and the Debian Package Maven plugin
 * (c) Copyright 2016 Gerrit Hohl
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package net.sourceforge.javadpkg.impl;

import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.util.List;

import net.sourceforge.javadpkg.Context;
import net.sourceforge.javadpkg.GlobalConstants;
import net.sourceforge.javadpkg.ParseException;
import net.sourceforge.javadpkg.Symbols;
import net.sourceforge.javadpkg.SymbolsParser;
import net.sourceforge.javadpkg.control.PackageDependency;
import net.sourceforge.javadpkg.control.PackageDependencyParser;
import net.sourceforge.javadpkg.control.PackageVersion;
import net.sourceforge.javadpkg.control.PackageVersionParser;
import net.sourceforge.javadpkg.control.impl.PackageDependencyParserImpl;
import net.sourceforge.javadpkg.control.impl.PackageNameParserImpl;
import net.sourceforge.javadpkg.control.impl.PackageVersionParserImpl;
import net.sourceforge.javadpkg.control.impl.PackageVersionRelationOperatorParserImpl;
import net.sourceforge.javadpkg.field.Field;
import net.sourceforge.javadpkg.field.FieldParser;
import net.sourceforge.javadpkg.field.impl.FieldParserImpl;
import net.sourceforge.javadpkg.io.DataSource;
import net.sourceforge.javadpkg.io.impl.DataStreamSource;


/**
 * <p>
 * A {@link SymbolsParser} implementation.
 * </p>
 *
 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
 * @version <b>1.0</b>, 10.01.2016 by Gerrit Hohl
 */
public class SymbolsParserImpl implements SymbolsParser, GlobalConstants {
	
	
	/** The parser for the package version. */
	private PackageVersionParser	packageVersionParser;
	/** The parser for the package dependencies. */
	private PackageDependencyParser	packageDependencyParser;
	/** The parser for the fields. */
	private FieldParser				fieldParser;


	/**
	 * <p>
	 * Creates a parser.
	 * </p>
	 */
	public SymbolsParserImpl() {
		super();

		this.packageVersionParser = new PackageVersionParserImpl();
		this.packageDependencyParser = new PackageDependencyParserImpl(new PackageNameParserImpl(),
				new PackageVersionRelationOperatorParserImpl(), this.packageVersionParser);
		this.fieldParser = new FieldParserImpl(false, false, false);
	}
	
	
	@Override
	public Symbols parseSymbols(DataSource source, Context context) throws IOException, ParseException {
		SymbolsImpl symbols = null;
		String line;
		boolean contentParsed = false;
		SymbolsEntryImpl entry = null;
		
		
		if (source == null)
			throw new IllegalArgumentException("Argument source is null.");
		if (context == null)
			throw new IllegalArgumentException("Argument context is null.");
		
		symbols = new SymbolsImpl();
		try {
			try (BufferedReader reader = new BufferedReader(new InputStreamReader(source.getInputStream(), UTF_8_CHARSET))) {
				while ((line = reader.readLine()) != null) {
					// --- Alternative dependency ---
					if (line.startsWith("| ")) {
						if (entry == null)
							throw new ParseException("Symbol line |" + line + "| of symbols from source |" + source.getName()
									+ "| contains an alternative dependency template also no initial line of an entry has been read.");
						if (contentParsed)
							throw new ParseException("Symbol line |" + line + "| of symbols from source |" + source.getName()
									+ "| contains an alternative dependency template also symbols have been already read.");
						this.parseAlternativeDependencyTemplate(source, entry, line);
					}
					// --- Symbol ---
					else if (line.startsWith(" ")) {
						if (entry == null)
							throw new ParseException("Symbol line |" + line + "| of symbols from source |" + source.getName()
									+ "| contains a symbol also no initial line of an entry has been read.");
						this.parseSymbol(source, context, entry, line);
						contentParsed = true;
					}
					// --- Meta data fields --
					else if (line.startsWith("* ")) {
						if (entry == null)
							throw new ParseException("Symbol line |" + line + "| of symbols from source |" + source.getName()
									+ "| contains a meta data field also no initial line of an entry has been read.");
						this.parseMetaData(source, context, entry, line);
						contentParsed = true;
					}
					// --- Initial line of an entry ---
					else {
						entry = this.parseInitialLine(source, context, line);
						symbols.addEntry(entry);
						contentParsed = false;
					}
				}
			}
		} catch (IOException e) {
			throw new IOException(
					"Couldn't read the shared libraries from source |" + source.getName() + "|: " + e.getMessage());
		}
		if (entry == null)
			throw new ParseException("The shared libraries from source |" + source.getName() + "| were empty.");
		return symbols;
	}
	
	
	/**
	 * <p>
	 * Parses the initial line.
	 * </p>
	 *
	 * @param source
	 *            The source.
	 * @param context
	 *            The context.
	 * @param line
	 *            The initial line.
	 * @return The symbols.
	 * @throws ParseException
	 *             If an error occurs during parsing.
	 */
	private SymbolsEntryImpl parseInitialLine(DataSource source, Context context, String line) throws ParseException {
		SymbolsEntryImpl entry;
		String[] parts;
		String libraryName, dependencyTemplate;
		PackageDependency dependency = null;
		
		
		parts = line.split(" ", 2);
		if (parts.length != 2)
			throw new ParseException("Couldn't split line |" + line + "| of symbols from source |" + source.getName()
					+ "| into 3 parts. Got " + parts.length);
		libraryName = parts[0];
		dependencyTemplate = parts[1];
		if (!dependencyTemplate.endsWith("#MINVER#")) {
			try {
				dependency = this.packageDependencyParser.parsePackageDependency(dependencyTemplate, context);
			} catch (ParseException e) {
				throw new ParseException("Couldn't parse dependency |" + dependencyTemplate + "| from |" + line
						+ "| of symbols from source |" + source.getName() + "| into 3 parts. Got " + parts.length);
			}
		}
		
		if (dependency == null) {
			entry = new SymbolsEntryImpl(libraryName, dependencyTemplate);
		} else {
			entry = new SymbolsEntryImpl(libraryName, dependency);
		}
		return entry;
	}


	/**
	 * <p>
	 * Parses the alternative dependency template.
	 * </p>
	 *
	 * @param source
	 *            The source.
	 * @param entry
	 *            The entry.
	 * @param line
	 *            The line.
	 * @throws ParseException
	 *             If an error occurs during parsing.
	 */
	private void parseAlternativeDependencyTemplate(DataSource source, SymbolsEntryImpl entry, String line)
			throws ParseException {
		
		String alternativeDependencyTemplate;

		// TODO Implement the parsing of the dependencies.
		alternativeDependencyTemplate = line.substring(2);
		entry.setAlternativeDependencyValue(alternativeDependencyTemplate);
	}
	
	
	/**
	 * <p>
	 * Parses a symbol.
	 * </p>
	 *
	 * @param source
	 *            The source.
	 * @param context
	 *            The context.
	 * @param entry
	 *            The entry.
	 * @param line
	 *            The line.
	 * @throws ParseException
	 *             If an error occurs during parsing.
	 */
	private void parseSymbol(DataSource source, Context context, SymbolsEntryImpl entry, String line) throws ParseException {
		String[] parts;
		PackageVersion version;
		String symbolName, symbolVersion;


		parts = line.trim().split(" ", 3);
		if ((parts.length < 2) || (parts.length > 3))
			throw new ParseException("Couldn't split line |" + line + "| of symbols from source |" + source.getName()
					+ "| into 2 or 3 parts. Got " + parts.length);
		
		try {
			version = this.packageVersionParser.parsePackageVersion(parts[1], context);
		} catch (ParseException e) {
			throw new ParseException("Couldn't parse version |" + parts[1] + "| from |" + line + "| of symbols from source |"
					+ source.getName() + "|: " + e.getMessage(), e);
		}
		parts = parts[0].split("@", 2);
		if (parts.length != 2)
			throw new ParseException("Couldn't split |" + line + "| of symbols from source |" + source.getName()
					+ "| into 2 parts. Got " + parts.length);
		symbolName = parts[0];
		symbolVersion = parts[1];
		
		entry.addSymbol(symbolName, symbolVersion, version);
	}
	
	
	/**
	 * <p>
	 * Parses a meta data field.
	 * </p>
	 *
	 * @param source
	 *            The source.
	 * @param context
	 *            The context.
	 * @param entry
	 *            The entry.
	 * @param line
	 *            The line.
	 * @throws ParseException
	 *             If an error occurs during parsing.
	 */
	private void parseMetaData(DataSource source, Context context, SymbolsEntryImpl entry, String line) throws ParseException {
		String trimmedLine;
		List<Field> fields;
		Field field;
		PackageDependency dependency;
		
		
		trimmedLine = line.substring(2);
		try {
			try (DataSource fieldSource = new DataStreamSource(new ByteArrayInputStream(trimmedLine.getBytes()), "field",
					false)) {
				fields = this.fieldParser.parseFieldsAsList(fieldSource, context);
			}
		} catch (IOException | ParseException e) {
			throw new ParseException("Couldn't parse meta data field in line |" + line + "| of symbols from source |"
					+ source.getName() + "|: " + e.getMessage(), e);
		}
		if (fields.isEmpty())
			throw new ParseException("Couldn't parse meta data field in line |" + line + "| of symbols from source |"
					+ source.getName() + "|: Didn't find any field.");
		field = fields.get(0);
		switch (field.getName().toLowerCase()) {
			case "build-depends-package":
				// TODO Check if the field is already set in the entry.
				try {
					dependency = this.packageDependencyParser.parsePackageDependency(field.getValue(), context);
				} catch (ParseException e) {
					throw new ParseException("Couldn't parse meta data field |" + field.getName() + "| in line |" + line
							+ "| of symbols from source |" + source.getName() + "|: " + e.getMessage(), e);
				}
				entry.setBuildDependsPackage(dependency);
				break;
			
			default:
				throw new ParseException("Found unsupported meta data field |" + field.getName() + "| in line |" + line
						+ "| of symbols from source |" + source.getName() + "|.");
		}
	}
	
	
}
