/*
 * dpkg - Debian Package library and the Debian Package Maven plugin
 * (c) Copyright 2016 Gerrit Hohl
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package net.sourceforge.javadpkg.impl;

import java.util.ArrayList;
import java.util.List;

import net.sourceforge.javadpkg.SymbolsEntry;
import net.sourceforge.javadpkg.control.PackageDependency;
import net.sourceforge.javadpkg.control.PackageVersion;


/**
 * <p>
 * A {@link SymbolsEntry} implementation.
 * </p>
 *
 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
 * @version <b>1.0</b>, 11.01.2016 by Gerrit Hohl
 */
public class SymbolsEntryImpl implements SymbolsEntry {
	
	
	/** The name of the library. */
	private String				libraryName;
	/** The dependency as string value. */
	private String				dependencyValue;
	/** The dependency. */
	private PackageDependency	dependency;

	/** The alternative dependency as string value. */
	private String				alternativeDependencyValue;
	/** The symbols. */
	private List<Symbol>		symbols;
	/** The build dependency. */
	private PackageDependency	buildDependsPackage;


	/**
	 * <p>
	 * Creates an entry.
	 * </p>
	 *
	 * @param libraryName
	 *            The name of the library.
	 * @throws IllegalArgumentException
	 *             If the name is <code>null</code>.
	 */
	private SymbolsEntryImpl(String libraryName) {
		super();
		
		if (libraryName == null)
			throw new IllegalArgumentException("Argument libraryName is null.");
			
		this.libraryName = libraryName;
		this.dependencyValue = null;
		this.dependency = null;

		this.alternativeDependencyValue = null;
		this.symbols = new ArrayList<>();
	}
	
	
	/**
	 * <p>
	 * Creates an entry.
	 * </p>
	 *
	 * @param libraryName
	 *            The name of the library.
	 * @param dependencyValue
	 *            The dependency as string value.
	 * @throws IllegalArgumentException
	 *             If any of the parameters are <code>null</code>.
	 */
	public SymbolsEntryImpl(String libraryName, String dependencyValue) {
		this(libraryName);

		if (dependencyValue == null)
			throw new IllegalArgumentException("Argument dependencyValue is null.");
			
		this.dependencyValue = dependencyValue;
	}


	/**
	 * <p>
	 * Creates an entry.
	 * </p>
	 *
	 * @param libraryName
	 *            The name of the library.
	 * @param dependency
	 *            The dependency.
	 * @throws IllegalArgumentException
	 *             If any of the parameters are <code>null</code>.
	 */
	public SymbolsEntryImpl(String libraryName, PackageDependency dependency) {
		this(libraryName);
		
		if (dependency == null)
			throw new IllegalArgumentException("Argument dependency is null.");

		this.dependency = dependency;
	}
	
	
	/**
	 * <p>
	 * Sets the alternative dependency as string value.
	 * </p>
	 *
	 * @param alternativeDependencyValue
	 *            The alternative dependency.
	 */
	public void setAlternativeDependencyValue(String alternativeDependencyValue) {
		this.alternativeDependencyValue = alternativeDependencyValue;
	}


	/**
	 * <p>
	 * Adds a symbol.
	 * </p>
	 *
	 * @param name
	 *            The name of the symbol.
	 * @param version
	 *            The version of the symbol.
	 * @param upstreamVersion
	 *            The upstream version.
	 */
	public void addSymbol(String name, String version, PackageVersion upstreamVersion) {
		Symbol symbol;
		
		
		if (name == null)
			throw new IllegalArgumentException("Argument name is null.");
		if (version == null)
			throw new IllegalArgumentException("Argument version is null.");
		if (upstreamVersion == null)
			throw new IllegalArgumentException("Argument upstreamVersion is null.");

		symbol = new Symbol(name, version, upstreamVersion);
		this.symbols.add(symbol);
	}
	
	
	/**
	 * <p>
	 * Sets the build dependency.
	 * </p>
	 *
	 * @param buildDependsPackage
	 *            The build dependency.
	 */
	public void setBuildDependsPackage(PackageDependency buildDependsPackage) {
		this.buildDependsPackage = buildDependsPackage;
	}
	
	
	@Override
	public String getText() {
		StringBuilder sb;
		
		
		sb = new StringBuilder();
		sb.append(this.libraryName);
		sb.append(' ');
		if (this.dependency == null)
			sb.append(this.dependencyValue);
		else {
			// TODO Need a PackageDependencyTransformer.
		}
		if (this.alternativeDependencyValue != null) {
			sb.append("\n| ");
			sb.append(this.alternativeDependencyValue);
		}
		if (this.symbols.isEmpty())
			sb.append('\n');
		else {
			for (Symbol symbol : this.symbols) {
				sb.append("\n ");
				sb.append(symbol.getName());
				sb.append('@');
				sb.append(symbol.getVersion());
				sb.append(' ');
				sb.append(symbol.getUpstreamVersion().getText());
			}
		}
		if (this.buildDependsPackage != null) {
			// TODO Need a PackageDependencyTransformer.
		}
		return sb.toString();
	}


	/* **********************************************************************
	 * **********************************************************************
	 * **********************************************************************
	 * **********************************************************************
	 * **********************************************************************
	 */
	
	
	/**
	 * <p>
	 * A symbol.
	 * </p>
	 *
	 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
	 * @version <b>1.0</b>, 11.01.2016 by Gerrit Hohl
	 */
	private class Symbol {
		
		
		/** The name of the symbol. */
		private String			name;
		/** The version of the symbol. */
		private String			version;
		/** The upstream version. */
		private PackageVersion	upstreamVersion;


		/**
		 * <p>
		 * Creates a symbol.
		 * </p>
		 *
		 * @param name
		 *            The name of the symbol.
		 * @param version
		 *            The version of the symbol.
		 * @param upstreamVersion
		 *            The upstream version.
		 */
		public Symbol(String name, String version, PackageVersion upstreamVersion) {
			super();
			
			this.name = name;
			this.version = version;
			this.upstreamVersion = upstreamVersion;
		}


		/**
		 * <p>
		 * Returns the name of the symbol.
		 * </p>
		 *
		 * @return The name.
		 */
		public String getName() {
			return this.name;
		}


		/**
		 * <p>
		 * Returns the version of the symbol.
		 * </p>
		 *
		 * @return The version.
		 */
		public String getVersion() {
			return this.version;
		}


		/**
		 * <p>
		 * Returns the upstream version.
		 * </p>
		 *
		 * @return The upstream version.
		 */
		public PackageVersion getUpstreamVersion() {
			return this.upstreamVersion;
		}


	}
	
	
}
