/*
 * dpkg - Debian Package library and the Debian Package Maven plugin
 * (c) Copyright 2016 Gerrit Hohl
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package net.sourceforge.javadpkg.impl;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;

import net.sourceforge.javadpkg.Context;
import net.sourceforge.javadpkg.GlobalConstants;
import net.sourceforge.javadpkg.MD5Sums;
import net.sourceforge.javadpkg.MD5SumsParser;
import net.sourceforge.javadpkg.ParseException;
import net.sourceforge.javadpkg.io.DataSource;


/**
 * <p>
 * A {@link MD5SumsParser} implementation.
 * </p>
 *
 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
 * @version <b>1.0</b>, 03.01.2016 by Gerrit Hohl
 */
public class MD5SumsParserImpl implements MD5SumsParser, GlobalConstants {


	/**
	 * <p>
	 * Creates a parser.
	 * </p>
	 */
	public MD5SumsParserImpl() {
		super();
	}


	@Override
	public MD5Sums parseMD5Sums(DataSource source, Context context) throws IOException, ParseException {
		MD5SumsImpl md5Sums;
		String line, md5sum, path;
		String[] parts;
		
		
		if (source == null)
			throw new IllegalArgumentException("Argument source is null.");
		if (context == null)
			throw new IllegalArgumentException("Argument context is null.");

		md5Sums = new MD5SumsImpl();
		try {
			try (BufferedReader reader = new BufferedReader(new InputStreamReader(source.getInputStream(), UTF_8_CHARSET))) {
				while ((line = reader.readLine()) != null) {
					parts = line.split(" ", 2);
					if (parts.length != 2)
						throw new ParseException(
								"The line |" + line + "| doesn't consist of a MD5 sum part and a file path part.");

					// --- Get MD5 sum ---
					md5sum = parts[0];
					if (!md5sum.matches("[a-f0-9]{32}"))
						throw new ParseException(
								"The MD5 sums |" + md5sum + "| of the line |" + line + "| is not a valid MD5 sum.");
					// --- Get file path ---
					path = parts[1];

					// --- Add the entry ---
					md5Sums.addMD5Sum(md5sum, path);
				}
			}
		} catch (IOException e) {
			throw new IOException("Couldn't read MD5 sums from source |" + source.getName() + "|: " + e.getMessage());
		}
		return md5Sums;
	}
	
	
}
