/*
 * dpkg - Debian Package library and the Debian Package Maven plugin
 * (c) Copyright 2016 Gerrit Hohl
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package net.sourceforge.javadpkg.impl;

import java.io.BufferedWriter;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.List;

import net.sourceforge.javadpkg.BuildException;
import net.sourceforge.javadpkg.GlobalConstants;
import net.sourceforge.javadpkg.MD5SumsBuilder;
import net.sourceforge.javadpkg.io.DataTarget;
import net.sourceforge.javadpkg.store.DataStore;
import net.sourceforge.javadpkg.store.FileHash;


/**
 * <p>
 * A {@link MD5SumsBuilder} implementation.
 * </p>
 *
 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
 * @version <b>1.0</b>, 26.04.2016 by Gerrit Hohl
 * @version <b>1.1</b>, 23.01.2018 by Gerrit Hohl
 */
public class MD5SumsBuilderImpl implements MD5SumsBuilder, GlobalConstants {
	
	
	/**
	 * <p>
	 * Creates a builder.
	 * </p>
	 */
	public MD5SumsBuilderImpl() {
		super();
	}
	
	
	@Override
	public void buildMD5Sums(DataStore store, DataTarget target) throws IOException, BuildException {
		MessageDigest digest;
		List<FileHash> fileHashes;
		String path;


		if (store == null)
			throw new IllegalArgumentException("Argument store is null.");
		if (target == null)
			throw new IllegalArgumentException("Argument target is null.");

		// --- Create the digest ---
		try {
			digest = MessageDigest.getInstance("MD5");
		} catch (NoSuchAlgorithmException e) {
			throw new BuildException("Couldn't build MD5 sums: MD5 is not supported: " + e.getMessage(), e);
		}

		// --- Create hashes ---
		try {
			fileHashes = store.createFileHashes(digest);
		} catch (IOException e) {
			throw new IOException("Couldn't build MD5 sums: Couldn't create hashes: " + e.getMessage(), e);
		}

		// --- Write hashes ---
		try {
			try (BufferedWriter out = new BufferedWriter(new OutputStreamWriter(target.getOutputStream(), UTF_8_CHARSET))) {
				for (FileHash fileHash : fileHashes) {
					out.write(fileHash.getHashAsHex());
					out.write(" ");

					path = fileHash.getPath();
					if (path.startsWith("/")) {
						path = path.substring(1);
					}
					out.write(path);
					out.write("\n");
				}
			}
		} catch (IOException e) {
			throw new IOException("Couldn't build MD5 sums: Couldn't write MD5 sums: " + e.getMessage(), e);
		}
	}


}
