/*
 * dpkg - Debian Package library and the Debian Package Maven plugin
 * (c) Copyright 2016 Gerrit Hohl
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package net.sourceforge.javadpkg.impl;

import net.sourceforge.javadpkg.DebianPackageConstants;
import net.sourceforge.javadpkg.DocumentPaths;
import net.sourceforge.javadpkg.control.PackageName;


/**
 * <p>
 * A {@link DocumentPaths} implementation.
 * </p>
 *
 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
 * @version <b>1.0</b>, 06.05.2016 by Gerrit Hohl
 */
public class DocumentPathsImpl implements DocumentPaths, DebianPackageConstants {
	
	
	/** The base path of the document folder. */
	private String	documentBasePath;
	/** The path of the document folder of the Debian package. */
	private String	documentPath;
	/** The path of the copyright file. */
	private String	copyrightPath;
	/** The path of the change log file. */
	private String	changeLogPath;
	/** The path of the Debian change log file. */
	private String	changeLogDebianPath;
	/** The path of the GZIP compressed change log file. */
	private String	changeLogGzipPath;
	/** The path of the HTML formatted change log file. */
	private String	changeLogHtmlPath;
	/** The path of the GZIP compressed HTML formatted change log file. */
	private String	changeLogHtmlGzipPath;


	/**
	 * <p>
	 * Creates the paths.
	 * </p>
	 *
	 * @param name
	 *            The name of the package.
	 * @throws IllegalArgumentException
	 *             If the name is <code>null</code>.
	 */
	public DocumentPathsImpl(PackageName name) {
		super();
		
		if (name == null)
			throw new IllegalArgumentException("Argument name is null.");
		
		this.initialize(DOC_BASE_PATH, name);
	}


	/**
	 * <p>
	 * Initializes the paths.
	 * </p>
	 *
	 * @param docBasePath
	 *            The base of the documentation path.
	 * @param name
	 *            The name of the package.
	 */
	private void initialize(String docBasePath, PackageName name) {
		this.documentBasePath = docBasePath;
		this.documentPath = docBasePath + name.getName() + "/";
		this.copyrightPath = this.documentPath + "copyright";
		this.changeLogPath = this.documentPath + "changelog";
		this.changeLogDebianPath = this.changeLogPath + ".Debian";
		this.changeLogGzipPath = this.changeLogPath + ".gz";
		this.changeLogHtmlPath = this.changeLogPath + ".html";
		this.changeLogHtmlGzipPath = this.changeLogHtmlPath + ".gz";
	}


	@Override
	public String getDocumentBasePath() {
		return this.documentBasePath;
	}


	@Override
	public String getDocumentPath() {
		return this.documentPath;
	}
	
	
	@Override
	public boolean isCopyrightPath(String path) {
		if (path == null)
			throw new IllegalArgumentException("Argument path is null.");
		
		return this.copyrightPath.equals(path);
	}


	@Override
	public String getCopyrightPath() {
		return this.copyrightPath;
	}


	@Override
	public boolean isChangeLogPath(String path) {
		if (path == null)
			throw new IllegalArgumentException("Argument path is null.");
		
		return (path.equals(this.changeLogPath) || path.startsWith(this.changeLogPath + "."));
	}


	@Override
	public boolean isChangeLogDebianPath(String path) {
		if (path == null)
			throw new IllegalArgumentException("Argument path is null.");
		
		return (path.equals(this.changeLogDebianPath) || path.startsWith(this.changeLogDebianPath + "."));
	}


	@Override
	public boolean isChangeLogGzipPath(String path) {
		if (path == null)
			throw new IllegalArgumentException("Argument path is null.");
		
		if (!this.isChangeLogPath(path))
			return false;
		return path.endsWith(".gz");
	}


	@Override
	public boolean isChangeLogHtmlPath(String path) {
		int index;
		String name, extension;
		
		
		if (path == null)
			throw new IllegalArgumentException("Argument path is null.");
		
		if (!this.isChangeLogPath(path))
			return false;
		index = path.lastIndexOf('/');
		name = path.substring(index + 1);
		if (name.isEmpty())
			return false;
		index = name.indexOf('.');
		if (index == -1)
			return false;
		extension = name.substring(index + 1);
		if (extension.isEmpty())
			return false;
		
		return ("html".equals(extension) || extension.startsWith("html."));
	}


	@Override
	public String getChangeLogPath() {
		return this.changeLogPath;
	}


	@Override
	public String getChangeLogGzipPath() {
		return this.changeLogGzipPath;
	}


	@Override
	public String getChangeLogHtmlPath() {
		return this.changeLogHtmlPath;
	}


	@Override
	public String getChangeLogHtmlGzipPath() {
		return this.changeLogHtmlGzipPath;
	}


}
