/*
 * dpkg - Debian Package library and the Debian Package Maven plugin
 * (c) Copyright 2016 Gerrit Hohl
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package net.sourceforge.javadpkg.impl;

import java.io.BufferedWriter;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.util.HashMap;
import java.util.Map;

import org.apache.commons.compress.archivers.ar.ArArchiveEntry;
import org.apache.commons.compress.archivers.ar.ArArchiveOutputStream;
import org.apache.commons.compress.archivers.tar.TarArchiveEntry;
import org.apache.commons.compress.archivers.tar.TarArchiveOutputStream;
import org.apache.commons.compress.compressors.CompressorOutputStream;
import org.apache.commons.compress.compressors.gzip.GzipCompressorOutputStream;
import org.apache.commons.compress.compressors.gzip.GzipParameters;

import net.sourceforge.javadpkg.BuildException;
import net.sourceforge.javadpkg.ChangeLog;
import net.sourceforge.javadpkg.ChangeLogBuilder;
import net.sourceforge.javadpkg.Context;
import net.sourceforge.javadpkg.Copyright;
import net.sourceforge.javadpkg.CopyrightBuilder;
import net.sourceforge.javadpkg.DebianPackageBuilder;
import net.sourceforge.javadpkg.DebianPackageConstants;
import net.sourceforge.javadpkg.DocumentPaths;
import net.sourceforge.javadpkg.MD5SumsBuilder;
import net.sourceforge.javadpkg.Script;
import net.sourceforge.javadpkg.ScriptBuilder;
import net.sourceforge.javadpkg.ScriptVariableReplacer;
import net.sourceforge.javadpkg.control.BinaryControl;
import net.sourceforge.javadpkg.control.Control;
import net.sourceforge.javadpkg.control.ControlBuilder;
import net.sourceforge.javadpkg.control.PackageVersion;
import net.sourceforge.javadpkg.control.Size;
import net.sourceforge.javadpkg.control.impl.ControlBuilderImpl;
import net.sourceforge.javadpkg.io.DataSource;
import net.sourceforge.javadpkg.io.DataSwap;
import net.sourceforge.javadpkg.io.DataTarget;
import net.sourceforge.javadpkg.io.FileMode;
import net.sourceforge.javadpkg.io.FileOwner;
import net.sourceforge.javadpkg.io.Streams;
import net.sourceforge.javadpkg.io.impl.DataStreamTarget;
import net.sourceforge.javadpkg.io.impl.DataTempFileSwap;
import net.sourceforge.javadpkg.replace.ReplacementException;
import net.sourceforge.javadpkg.replace.Replacements;
import net.sourceforge.javadpkg.replace.ReplacementsMap;
import net.sourceforge.javadpkg.store.DataStore;
import net.sourceforge.javadpkg.store.DataStoreImpl;


/**
 * <p>
 * A {@link DebianPackageBuilder} implementation.
 * </p>
 *
 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
 * @version <b>1.0</b>, 25.04.2016 by Gerrit Hohl
 */
public class DebianPackageBuilderImpl implements DebianPackageBuilder, DebianPackageConstants {
	
	
	/**
	 * <p>
	 * The Debian package file format version written by this builder
	 * implementation.
	 * </p>
	 */
	private static final String		FILE_FORMAT_VERSION	= "2.0";
	
	
	/**
	 * <p>
	 * The default script which will be executed before the package is
	 * installed.
	 * </p>
	 */
	private Script					defaultPreInstall;
	/**
	 * <p>
	 * The default script which will be executed after the package is installed.
	 * </p>
	 */
	private Script					defaultPostInstall;
	/**
	 * <p>
	 * The default script which will be executed before the package is removed.
	 * </p>
	 */
	private Script					defaultPreRemove;
	/**
	 * <p>
	 * The default script which will be executed after the package is removed.
	 * </p>
	 */
	private Script					defaultPostRemove;
	
	/** The builder for the control. */
	private ControlBuilder			controlBuilder;
	/** The builder for the MD5 sums. */
	private MD5SumsBuilder			md5SumsBuilder;
	/** The builder for the scripts. */
	private ScriptBuilder			scriptBuilder;
	/** The variable replacer for the scripts. */
	private ScriptVariableReplacer	scriptVariableReplacer;
	/** The builder for the copyright. */
	private CopyrightBuilder		copyrightBuilder;
	/** The builder for the change log. */
	private ChangeLogBuilder		changeLogBuilder;
	
	/** The control information. */
	private Control					control;
	/** The overhead which will be added to the specified installed size. */
	private Size					installedSizeOverhead;
	/** The script which will be executed before the package is installed. */
	private Script					preInstall;
	/** The script which will be executed after the package is installed. */
	private Script					postInstall;
	/** The script which will be executed before the package is removed. */
	private Script					preRemove;
	/** The script which will be executed after the package is removed. */
	private Script					postRemove;
	/** The store for the data files. */
	private DataStore				dataStore;
	/** The copyright. */
	private Copyright				copyright;
	/** The change log. */
	private ChangeLog				changeLog;
	
	
	/**
	 * <p>
	 * Creates a builder.
	 * </p>
	 *
	 * @param defaultPreInstall
	 *            The default script which will be executed before the package
	 *            is installed.
	 * @param defaultPostInstall
	 *            The default script which will be executed after the package is
	 *            installed.
	 * @param defaultPreRemove
	 *            The default script which will be executed before the package
	 *            is removed.
	 * @param defaultPostRemove
	 *            The default script which will be executed after the package is
	 *            removed.
	 * @throws IllegalArgumentException
	 *             If any of the parameters are <code>null</code>.
	 */
	protected DebianPackageBuilderImpl(Script defaultPreInstall, Script defaultPostInstall, Script defaultPreRemove,
			Script defaultPostRemove) {
		
		super();
		
		if (defaultPreInstall == null)
			throw new IllegalArgumentException("Argument defaultPreInstall is null.");
		if (defaultPostInstall == null)
			throw new IllegalArgumentException("Argument defaultPostInstall is null.");
		if (defaultPreRemove == null)
			throw new IllegalArgumentException("Argument defaultPreRemove is null.");
		if (defaultPostRemove == null)
			throw new IllegalArgumentException("Argument defaultPostRemove is null.");
		
		this.defaultPreInstall = defaultPreInstall;
		this.defaultPostInstall = defaultPostInstall;
		this.defaultPreRemove = defaultPreRemove;
		this.defaultPostRemove = defaultPostRemove;
		
		this.controlBuilder = new ControlBuilderImpl();
		this.md5SumsBuilder = new MD5SumsBuilderImpl();
		this.scriptBuilder = new ScriptBuilderImpl();
		this.scriptVariableReplacer = new ScriptVariableReplacerImpl();
		this.copyrightBuilder = new CopyrightBuilderImpl();
		this.changeLogBuilder = new ChangeLogBuilderImpl();
		
		this.control = null;
		this.installedSizeOverhead = null;
		this.preInstall = null;
		this.postInstall = null;
		this.preRemove = null;
		this.postRemove = null;
		this.dataStore = new DataStoreImpl();
		this.copyright = null;
		this.changeLog = null;
	}
	
	
	@Override
	public void setControl(Control control) {
		this.control = control;
	}
	
	
	@Override
	public void setInstalledSizeOverhead(Size installedSizeOverhead) {
		this.installedSizeOverhead = installedSizeOverhead;
	}
	
	
	@Override
	public void setPreInstall(Script preInstall) {
		this.preInstall = preInstall;
	}
	
	
	@Override
	public void setPostInstall(Script postInstall) {
		this.postInstall = postInstall;
	}
	
	
	@Override
	public void setPreRemove(Script preRemove) {
		this.preRemove = preRemove;
	}
	
	
	@Override
	public void setPostRemove(Script postRemove) {
		this.postRemove = postRemove;
	}
	
	
	@Override
	public void addDataDirectory(String path) {
		if (path == null)
			throw new IllegalArgumentException("Argument path is null.");
		
		this.dataStore.addDirectory(path);
	}


	@Override
	public void addDataDirectory(String path, FileOwner owner, FileMode mode) {
		if (path == null)
			throw new IllegalArgumentException("Argument path is null.");
		if (owner == null)
			throw new IllegalArgumentException("Argument owner is null.");
		if (mode == null)
			throw new IllegalArgumentException("Argument mode is null.");
		
		this.dataStore.addDirectory(path, owner, mode);
	}


	@Override
	public void addDataFile(DataSource source, String path) {
		if (source == null)
			throw new IllegalArgumentException("Argument source is null.");
		if (path == null)
			throw new IllegalArgumentException("Argument path is null.");
		
		this.dataStore.addFile(source, path);
	}


	@Override
	public void addDataFile(DataSource source, String path, FileOwner owner, FileMode mode) {
		
		if (source == null)
			throw new IllegalArgumentException("Argument source is null.");
		if (path == null)
			throw new IllegalArgumentException("Argument path is null.");
		if (owner == null)
			throw new IllegalArgumentException("Argument owner is null.");
		if (mode == null)
			throw new IllegalArgumentException("Argument mode is null.");
		
		this.dataStore.addFile(source, path, owner, mode);
	}
	
	
	@Override
	public void addDataSymLink(String path, String target, FileOwner owner, FileMode mode) {
		
		if (path == null)
			throw new IllegalArgumentException("Argument path is null.");
		if (target == null)
			throw new IllegalArgumentException("Argument target is null.");
		if (owner == null)
			throw new IllegalArgumentException("Argument owner is null.");
		if (mode == null)
			throw new IllegalArgumentException("Argument mode is null.");
		
		this.dataStore.addSymLink(path, target, owner, mode);
	}
	
	
	@Override
	public void setCopyright(Copyright copyright) {
		this.copyright = copyright;
	}


	@Override
	public void setChangeLog(ChangeLog changeLog) {
		this.changeLog = changeLog;
	}
	
	
	@Override
	public void buildDebianPackage(DataTarget target, Context context) throws IOException, BuildException {
		
		
		if (target == null)
			throw new IllegalArgumentException("Argument target is null.");
		if (context == null)
			throw new IllegalArgumentException("Argument context is null.");
		
		if (this.control == null)
			throw new IllegalStateException("Can't build Debian package because no control information is set.");
		if (this.copyright == null)
			throw new IllegalStateException("Can't build Debian package because no copyright is set.");
		if (this.changeLog == null)
			throw new IllegalStateException("Can't build Debian package because no change log is set.");
		
		// TODO Perform a consistency check of the configuration of the Debian package.
		try (OutputStream out = target.getOutputStream()) {
			try (ArArchiveOutputStream arOut = new ArArchiveOutputStream(out)) {
				// --- Write the version of the package ---
				this.writeVersion(target, arOut);

				// --- Write the meta information ---
				this.writeControl(target, context, arOut);

				// --- Write the contents ---
				this.writeData(target, context, arOut);
			}
		}
	}


	/**
	 * <p>
	 * Writes the version of the Debian package.
	 * </p>
	 *
	 * @param target
	 *            The target.
	 * @param out
	 *            The stream on the archive.
	 * @throws IOException
	 *             If an I/O error occurs.
	 */
	private void writeVersion(DataTarget target, ArArchiveOutputStream out) throws IOException {
		ByteArrayOutputStream arrayOut;
		byte[] data;
		ArArchiveEntry entry;


		try {
			// --- Prepare the data (we need the exact length) ---
			arrayOut = new ByteArrayOutputStream();
			try (BufferedWriter writer = new BufferedWriter(new OutputStreamWriter(arrayOut, UTF_8_CHARSET))) {
				writer.write(FILE_FORMAT_VERSION);
				writer.write('\n');
			}
			data = arrayOut.toByteArray();
			
			// --- Write the entry (default flags are okay: UID: 0; GID: 0; MODE: 33188) ---
			entry = new ArArchiveEntry("debian-binary", data.length);
			out.putArchiveEntry(entry);
			out.write(data);
			out.closeArchiveEntry();
		} catch (IOException e) {
			throw new IOException("Couldn't write |debian-binary| in AR archive |" + target.getName() + "|: " + e.getMessage(),
					e);
		}
	}


	/**
	 * <p>
	 * Writes the meta information of the Debian package.
	 * </p>
	 *
	 * @param target
	 *            The target.
	 * @param context
	 *            The context.
	 * @param out
	 *            The stream on the archive.
	 * @throws IOException
	 *             If an I/O error occurs.
	 * @throws BuildException
	 *             If an error occurs during the building.
	 */
	private void writeControl(DataTarget target, Context context, ArArchiveOutputStream out)
			throws IOException, BuildException {
		
		File controlFile;
		Size installedSize = null;
		PackageVersion version;
		boolean deleted = false;
		
		
		// --- Get the installed size of the package (from the control or by counting) ---
		if (this.control instanceof BinaryControl) {
			installedSize = ((BinaryControl) this.control).getInstalledSize();
		}
		if (installedSize == null) {
			try {
				installedSize = this.dataStore.getSize();
			} catch (IOException e) {
				throw new IOException(
						"Couldn't determine size of all files which should be added to the Debian package: " + e.getMessage(),
						e);
			}
		}
		// --- Add the overhead (if available) ---
		if (this.installedSizeOverhead != null) {
			installedSize = new Size(installedSize.getBytes() + this.installedSizeOverhead.getBytes());
		}

		/*
		 * --- Write the control file.
		 *     Unfortunately we can't write it directly into the AR archive because of the
		 *     commons-compress implementation which needs the size before storing the entry. ---
		 */
		controlFile = File.createTempFile(CONTROL_NAME, TAR_GZIP_SUFFIX);
		try {
			try (FileOutputStream fileOut = new FileOutputStream(controlFile)) {
				try (TarArchiveOutputStream tarOut = this.createTarArchive(fileOut)) {
					// --- Root entry ---
					this.writeTarEntry(tarOut, "", ROOT_GROUP_ID, ROOT_GROUP_NAME, ROOT_USER_ID, ROOT_USER_NAME, DIRECTORY_MODE,
							null);
					
					// --- Control information ---
					try (DataSwap swap = new DataTempFileSwap(CONTROL_ENTRY)) {
						this.controlBuilder.buildControl(this.control, installedSize, swap.getTarget(), context);
						this.writeTarEntry(tarOut, CONTROL_ENTRY, ROOT_GROUP_ID, ROOT_GROUP_NAME, ROOT_USER_ID, ROOT_USER_NAME,
								FILE_MODE, swap.getSource());
					}
					
					// --- MD5 sums ---
					try (DataSwap swap = new DataTempFileSwap(MD5SUMS_ENTRY)) {
						this.md5SumsBuilder.buildMD5Sums(this.dataStore, swap.getTarget());
						this.writeTarEntry(tarOut, MD5SUMS_ENTRY, ROOT_GROUP_ID, ROOT_GROUP_NAME, ROOT_USER_ID, ROOT_USER_NAME,
								FILE_MODE, swap.getSource());
					}
					
					
					// --- Scripts are only used for binary packages ---
					if (this.control instanceof BinaryControl) {
						version = ((BinaryControl) this.control).getVersion();
						
						// --- Pre-installation script ---
						this.writeScript(tarOut, this.preInstall, this.defaultPreInstall, PREINST_ENTRY, version, context);
						
						// --- Post-installation script ---
						this.writeScript(tarOut, this.postInstall, this.defaultPostInstall, POSTINST_ENTRY, version, context);
						
						// --- Pre-removal script ---
						this.writeScript(tarOut, this.preRemove, this.defaultPreRemove, PRERM_ENTRY, version, context);
						
						// --- Post-removal script ---
						this.writeScript(tarOut, this.postRemove, this.defaultPostRemove, POSTRM_ENTRY, version, context);
					}
					
					// --- Templates ---
					// TODO Write templates.
					//					this.createTarEntry(tarOut, TEMPLATES_ENTRY, ROOT_GROUP_ID, ROOT_GROUP_NAME, ROOT_USER_ID, ROOT_USER_NAME,
					//							FILE_MODE);
					//					tarOut.closeArchiveEntry();
					
					// --- Configuration ---
					// TODO Write configuration.
					//					this.createTarEntry(tarOut, CONFIG_ENTRY, ROOT_GROUP_ID, ROOT_GROUP_NAME, ROOT_USER_ID, ROOT_USER_NAME,
					//							SCRIPT_MODE);
					//					tarOut.closeArchiveEntry();
					
					// --- Configuration files ---
					// TODO Write configuration files.
					//					this.createTarEntry(tarOut, CONFFILES_ENTRY, ROOT_GROUP_ID, ROOT_GROUP_NAME, ROOT_USER_ID, ROOT_USER_NAME,
					//							FILE_MODE);
					//					tarOut.closeArchiveEntry();
					
					// --- Shared libraries ---
					// TODO Write shared libraries.
					//					this.createTarEntry(tarOut, SHLIBS_ENTRY, ROOT_GROUP_ID, ROOT_GROUP_NAME, ROOT_USER_ID, ROOT_USER_NAME,
					//							FILE_MODE);
					//					tarOut.closeArchiveEntry();
					
					// --- Symbols ---
					// TODO Write symbols.
					//					this.createTarEntry(tarOut, SYMBOLS_ENTRY, ROOT_GROUP_ID, ROOT_GROUP_NAME, ROOT_USER_ID, ROOT_USER_NAME,
					//							FILE_MODE);
					//					tarOut.closeArchiveEntry();
					
					// --- Triggers ---
					// TODO Write triggers.
					//					this.createTarEntry(tarOut, TRIGGERS_ENTRY, ROOT_GROUP_ID, ROOT_GROUP_NAME, ROOT_USER_ID, ROOT_USER_NAME,
					//							FILE_MODE);
					//					tarOut.closeArchiveEntry();
				}
			}
			
			// --- Write the entry (default flags are okay: UID: 0; GID: 0; MODE: 33188) ---
			this.writeArEntry(out, CONTROL_NAME + TAR_GZIP_SUFFIX, controlFile);
		} finally {
			// --- Try to delete the control file ---
			if (controlFile.exists() && !controlFile.delete()) {
				deleted = false;
			} else {
				deleted = true;
			}
		}
		if (!deleted)
			throw new IOException("Couldn't delete temporary control file |" + controlFile.getAbsolutePath() + "|.");
	}


	/**
	 * <p>
	 * Writes an installation or removal script.
	 * </p>
	 *
	 * @param out
	 *            The stream.
	 * @param script
	 *            The script.
	 * @param defaultScript
	 *            The default script if the no script is passed.
	 * @param name
	 *            The name of the entry.
	 * @param version
	 *            The version.
	 * @param context
	 *            The context.
	 * @throws IOException
	 *             If an I/O error occurs.
	 * @throws BuildException
	 *             If an error occurs during the build.
	 */
	private void writeScript(TarArchiveOutputStream out, Script script, Script defaultScript, String name,
			PackageVersion version, Context context) throws IOException, BuildException {
		
		Script effectiveScript;
		Map<String, String> variables;
		Replacements replacements;
		
		
		if (script == null) {
			variables = new HashMap<>();
			// TODO Extract to constants. Maybe we should use Replacements from outside which includes the control properties.
			variables.put("deb.version", version.getText());
			replacements = new ReplacementsMap(variables);
			try {
				effectiveScript = this.scriptVariableReplacer.replaceScriptVariables(defaultScript, replacements, context);
			} catch (ReplacementException e) {
				throw new BuildException("Couldn't write default script |" + name + "|: " + e.getMessage(), e);
			}
		} else {
			effectiveScript = script;
		}

		try (DataSwap swap = new DataTempFileSwap(name)) {
			this.scriptBuilder.buildScript(swap.getTarget(), effectiveScript);
			this.writeTarEntry(out, name, ROOT_GROUP_ID, ROOT_GROUP_NAME, ROOT_USER_ID, ROOT_USER_NAME, SCRIPT_MODE,
					swap.getSource());
		}
	}


	/**
	 * <p>
	 * Writes the contents of the Debian package.
	 * </p>
	 *
	 * @param target
	 *            The target.
	 * @param context
	 *            The context.
	 * @param out
	 *            The stream on the archive.
	 * @throws IOException
	 *             If an I/O error occurs.
	 * @throws BuildException
	 *             If an error occurs during the build.
	 */
	@SuppressWarnings("resource")
	private void writeData(DataTarget target, Context context, ArArchiveOutputStream out) throws IOException, BuildException {
		BinaryControl control;
		DocumentPaths paths;
		String[] ensurePaths;
		File dataFile;
		boolean deleted = false;
		DataSwap swapCopyright = null, swapChangeLog = null;
		
		
		// --- Create the document paths ---
		if (this.control instanceof BinaryControl) {
			control = (BinaryControl) this.control;
		} else
			throw new BuildException("Found control |" + this.control + "| of type |"
					+ (this.control == null ? "null" : this.control.getClass().getCanonicalName())
					+ ", but only control of type |" + BinaryControl.class.getCanonicalName() + "| is supported.");
		paths = new DocumentPathsImpl(control.getPackage());
		
		
		// --- Make sure the copyright and the change log are not added directly ---
		// TODO Check for changelog.Debian, changelog.Debian.gz and so on.
		if (this.dataStore.exists(paths.getCopyrightPath()))
			throw new BuildException("Found |" + paths.getCopyrightPath()
					+ " added to the builder, but the copyright should be added through the corresponding method of the builder.");
		else if (this.dataStore.exists(paths.getChangeLogPath()))
			throw new BuildException("Found |" + paths.getChangeLogPath()
					+ " added to the builder, but the copyright should be added through the corresponding method of the builder.");
		else if (this.dataStore.exists(paths.getChangeLogGzipPath()))
			throw new BuildException("Found |" + paths.getChangeLogGzipPath()
					+ " added to the builder, but the copyright should be added through the corresponding method of the builder.");
		else if (this.dataStore.exists(paths.getChangeLogHtmlPath()))
			throw new BuildException("Found |" + paths.getChangeLogHtmlPath()
					+ " added to the builder, but the copyright should be added through the corresponding method of the builder.");
		else if (this.dataStore.exists(paths.getChangeLogHtmlGzipPath()))
			throw new BuildException("Found |" + paths.getChangeLogHtmlGzipPath()
					+ " added to the builder, but the copyright should be added through the corresponding method of the builder.");
		
		// --- Make sure the document folder for the Debian package exists ---
		ensurePaths = new String[] { USR_PATH, USR_SHARE_PATH, paths.getDocumentBasePath(), paths.getDocumentPath() };
		for (String ensurePath : ensurePaths) {
			if (!this.dataStore.exists(ensurePath)) {
				this.dataStore.addDirectory(ensurePath);
			}
		}
		
		/*
		 * --- Write the data file.
		 *     Unfortunately we can't write it directly into the AR archive because of the
		 *     commons-compress implementation which needs the size before storing the entry. ---
		 */
		dataFile = File.createTempFile(DATA_NAME, TAR_GZIP_SUFFIX);
		try {
			// --- Create the copyright ---
			swapCopyright = new DataTempFileSwap("copyright");
			this.copyrightBuilder.buildCopyright(this.copyright, swapCopyright.getTarget(), context);
			this.dataStore.addFile(swapCopyright.getSource(), paths.getCopyrightPath());
			
			try (DataSwap swap = new DataTempFileSwap("changelog")) {
				this.changeLogBuilder.buildChangeLog(this.changeLog, swap.getTarget(), context);
				swapChangeLog = new DataTempFileSwap("changelog.gz");
				Streams.compressGzip(swap.getSource(), swapChangeLog.getTarget(), 9);
			}
			this.dataStore.addFile(swapChangeLog.getSource(), paths.getChangeLogGzipPath());
			
			// --- Write all directories and files of the data store ---
			try (FileOutputStream fileOut = new FileOutputStream(dataFile)) {
				try (TarArchiveOutputStream tarOut = this.createTarArchive(fileOut)) {
					this.dataStore.write(tarOut);
				}
			}
			
			// --- Write the entry (default flags are okay: UID: 0; GID: 0; MODE: 33188) ---
			this.writeArEntry(out, DATA_NAME + TAR_GZIP_SUFFIX, dataFile);
		} finally {
			// --- Close change log ---
			swapChangeLog.close();
			
			// --- Close copyright ---
			swapCopyright.close();
			
			// --- Try to delete the control file ---
			if (dataFile.exists() && !dataFile.delete()) {
				deleted = false;
			} else {
				deleted = true;
			}
		}
		if (!deleted)
			throw new IOException("Couldn't delete temporary data file |" + dataFile.getAbsolutePath() + "|.");
	}
	
	
	/**
	 * <p>
	 * Creates TAR archive upon the specified stream.
	 * </p>
	 *
	 * @param out
	 *            The underlying stream.
	 * @return The stream on the TAR archive.
	 * @throws IOException
	 *             If an I/O error occurs while opening the TAR archive.
	 */
	@SuppressWarnings("resource")
	private TarArchiveOutputStream createTarArchive(OutputStream out) throws IOException {
		TarArchiveOutputStream tarOut;
		GzipParameters gzipParameters;
		CompressorOutputStream compressorOut;
		
		
		// ROADMAP Configurable compression (GZIP / BZIP2 / XZ).
		gzipParameters = new GzipParameters();
		gzipParameters.setCompressionLevel(9);
		compressorOut = new GzipCompressorOutputStream(out, gzipParameters);
		
		tarOut = new TarArchiveOutputStream(compressorOut);
		tarOut.setLongFileMode(TarArchiveOutputStream.LONGFILE_GNU);
		return tarOut;
	}
	
	
	/**
	 * <p>
	 * Writes a entry into the TAR archive.
	 * </p>
	 *
	 * @param out
	 *            The stream on the TAR archive.
	 * @param name
	 *            The name of the entry.
	 * @param groupId
	 *            The group ID.
	 * @param groupName
	 *            The group name.
	 * @param userId
	 *            The user ID.
	 * @param userName
	 *            The user name.
	 * @param mode
	 *            The mode.
	 * @param source
	 *            The source for the content of the entry (optional).
	 * @throws IOException
	 *             If an I/O error occurs.
	 */
	private void writeTarEntry(TarArchiveOutputStream out, String name, long groupId, String groupName, long userId,
			String userName, int mode, DataSource source) throws IOException {
		
		StringBuilder entryName;
		TarArchiveEntry tarEntry;
		
		
		// --- Create entry name ---
		entryName = new StringBuilder();
		if (!name.startsWith("./")) {
			entryName.append('.');
			if (!name.startsWith("/")) {
				entryName.append('/');
			}
		}
		entryName.append(name);
		
		// --- Open entry ---
		tarEntry = new TarArchiveEntry(entryName.toString());
		tarEntry.setGroupId(groupId);
		tarEntry.setGroupName(groupName);
		tarEntry.setUserId(userId);
		tarEntry.setUserName(userName);
		tarEntry.setMode(mode);
		if (source != null) {
			if (source.getLength() < 0)
				throw new IOException("Couldn't create entry |" + name
						+ "| in TAR archive stream: Couldn't determine size of source |" + source.getName() + "|.");
			tarEntry.setSize(source.getLength());
		}
		try {
			out.putArchiveEntry(tarEntry);
		} catch (IOException e) {
			throw new IOException("Couldn't create entry |" + name + "| in TAR archive stream: " + e.getMessage(), e);
		}

		// --- Copy the content ---
		if (source != null) {
			try {
				if (source.getLength() > 0) {
					try (DataTarget target = new DataStreamTarget(out, name, false)) {
						Streams.copy(source, target);
					}
				}
			} catch (IOException e) {
				throw new IOException("Couldn't create entry |" + name + "| in TAR archive stream: " + e.getMessage(), e);
			}
		}

		// --- Close entry ---
		try {
			out.closeArchiveEntry();
		} catch (IOException e) {
			throw new IOException("Couldn't create entry |" + name + "| in TAR archive stream: " + e.getMessage(), e);
		}
	}


	/**
	 * <p>
	 * Writes a entry into the AR archive.
	 * </p>
	 *
	 * @param out
	 *            The stream on the AR archive.
	 * @param name
	 *            The name of the entry.
	 * @param file
	 *            The file having the content of the entry.
	 * @throws IOException
	 *             If an I/O error occurs.
	 */
	private void writeArEntry(ArArchiveOutputStream out, String name, File file) throws IOException {
		ArArchiveEntry entry;


		try {
			entry = new ArArchiveEntry(name, file.length());
			out.putArchiveEntry(entry);
			try (InputStream in = Streams.createBufferedFileInputStream(file)) {
				Streams.copy(in, out);
			}
			out.closeArchiveEntry();
		} catch (IOException e) {
			throw new IOException("Couldn't write file |" + file.getAbsolutePath() + "| as entry |" + name
					+ "| into the archive stream: " + e.getMessage(), e);
		}
	}


}
