/*
 * dpkg - Debian Package library and the Debian Package Maven plugin
 * (c) Copyright 2016 Gerrit Hohl
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package net.sourceforge.javadpkg.impl;

import java.io.IOException;

import net.sourceforge.javadpkg.Context;
import net.sourceforge.javadpkg.DebianPackageBuilder;
import net.sourceforge.javadpkg.DebianPackageBuilderFactory;
import net.sourceforge.javadpkg.ParseException;
import net.sourceforge.javadpkg.Script;
import net.sourceforge.javadpkg.ScriptParser;
import net.sourceforge.javadpkg.io.DataSource;
import net.sourceforge.javadpkg.io.Streams;


/**
 * <p>
 * A {@link DebianPackageBuilderFactory} implementation.
 * </p>
 *
 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
 * @version <b>1.0</b>, 25.04.2016 by Gerrit Hohl
 */
public class DebianPackageBuilderFactoryImpl implements DebianPackageBuilderFactory {
	
	
	/** The parser for the scripts. */
	private ScriptParser	scriptParser;
							
	/**
	 * <p>
	 * The default script which will be executed before the package is
	 * installed.
	 * </p>
	 */
	private Script			defaultPreInstall;
	/**
	 * <p>
	 * The default script which will be executed after the package is installed.
	 * </p>
	 */
	private Script			defaultPostInstall;
	/**
	 * <p>
	 * The default script which will be executed before the package is removed.
	 * </p>
	 */
	private Script			defaultPreRemove;
	/**
	 * <p>
	 * The default script which will be executed after the package is removed.
	 * </p>
	 */
	private Script			defaultPostRemove;
							
							
	/**
	 * <p>
	 * Creates a factory.
	 * </p>
	 * 
	 * @throws IOException
	 *             If an I/O error occurs.
	 * @throws ParseException
	 *             If an error occurs during the parsing.
	 */
	public DebianPackageBuilderFactoryImpl() throws IOException, ParseException {
		super();

		this.scriptParser = new ScriptParserImpl();

		this.initialize();
	}
	
	
	/**
	 * <p>
	 * Initializes the factory.
	 * </p>
	 *
	 * @throws IOException
	 *             If an I/O error occurs.
	 * @throws ParseException
	 *             If an error occurs during the parsing.
	 */
	private void initialize() throws IOException, ParseException {
		Context context;
		
		
		context = new ContextImpl();
		try (DataSource source = Streams.createResourceDataSource(this.getClass(), "preinst")) {
			this.defaultPreInstall = this.scriptParser.parseScript(source, context);
		}
		try (DataSource source = Streams.createResourceDataSource(this.getClass(), "postinst")) {
			this.defaultPostInstall = this.scriptParser.parseScript(source, context);
		}
		try (DataSource source = Streams.createResourceDataSource(this.getClass(), "prerm")) {
			this.defaultPreRemove = this.scriptParser.parseScript(source, context);
		}
		try (DataSource source = Streams.createResourceDataSource(this.getClass(), "postrm")) {
			this.defaultPostRemove = this.scriptParser.parseScript(source, context);
		}
	}
	
	
	@Override
	public DebianPackageBuilder createDebianPackageBuilder() {
		DebianPackageBuilder builder;


		builder = new DebianPackageBuilderImpl(this.defaultPreInstall, this.defaultPostInstall, this.defaultPreRemove,
				this.defaultPostRemove);
		return builder;
	}


}
