/*
 * dpkg - Debian Package library and the Debian Package Maven plugin
 * (c) Copyright 2016 Gerrit Hohl
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package net.sourceforge.javadpkg.impl;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import net.sourceforge.javadpkg.Context;
import net.sourceforge.javadpkg.Copyright;
import net.sourceforge.javadpkg.CopyrightConstants;
import net.sourceforge.javadpkg.CopyrightLicense;
import net.sourceforge.javadpkg.CopyrightParser;
import net.sourceforge.javadpkg.FilesCopyright;
import net.sourceforge.javadpkg.ParseException;
import net.sourceforge.javadpkg.field.Field;
import net.sourceforge.javadpkg.field.FieldParser;
import net.sourceforge.javadpkg.field.impl.FieldParserImpl;
import net.sourceforge.javadpkg.io.DataSource;


/**
 * <p>
 * A {@link CopyrightParser} implementation.
 * </p>
 *
 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
 * @version <b>1.0</b>, 04.05.2016 by Gerrit Hohl
 * @version <b>1.1</b>, 21.06.2018 by Gerrit Hohl
 */
public class CopyrightParserImpl implements CopyrightParser, CopyrightConstants {
	
	
	/** The parser for the fields. */
	private FieldParser	fieldParser;
	/** The flag if the comparison of the field names should be strict. */
	private boolean		strictFieldNames;
	
	
	/**
	 * <p>
	 * Creates a parser.
	 * </p>
	 */
	public CopyrightParserImpl() {
		super();
		
		this.fieldParser = new FieldParserImpl(true, true, true);
		this.strictFieldNames = false;
	}


	@Override
	public Copyright parseCopyright(DataSource source, Context context) throws IOException, ParseException {
		Copyright copyright;
		List<Field> fields;


		if (source == null)
			throw new IllegalArgumentException("Argument source is null.");
		if (context == null)
			throw new IllegalArgumentException("Argument context is null.");
		
		// --- Get the fields from the source ---
		fields = this.parseFields(source, context);
		
		// --- Get the copyright from the fields ---
		copyright = this.parseCopyright(fields, context);

		return copyright;
	}
	
	
	/**
	 * <p>
	 * Parses the fields from the source.
	 * </p>
	 *
	 * @param source
	 *            The source.
	 * @param context
	 *            The context.
	 * @return The fields.
	 * @throws IOException
	 *             If an I/O error occurs.
	 * @throws ParseException
	 *             If an error occurs while parsing the fields.
	 */
	private List<Field> parseFields(DataSource source, Context context) throws IOException, ParseException {
		List<Field> fields;


		try {
			fields = this.fieldParser.parseFieldsAsList(source, context);
		} catch (IOException e) {
			throw new IOException("Couldn't parse fields of copyright: " + e.getMessage(), e);
		} catch (ParseException e) {
			throw new ParseException("Couldn't parse fields of copyright: " + e.getMessage(), e);
		}
		return fields;
	}
	
	
	/**
	 * <p>
	 * Parses the copyright from the specified fields.
	 * </p>
	 *
	 * @param fields
	 *            The fields.
	 * @param context
	 *            The context.
	 * @return The copyright.
	 * @throws ParseException
	 *             If an error occurs while parsing the fields.
	 */
	private Copyright parseCopyright(List<Field> fields, Context context) throws ParseException {
		CopyrightImpl copyright = null;
		boolean header = true;
		CopyrightLicenseImpl license = null;
		List<FilesCopyright> filesCopyrights;
		FilesCopyrightImpl filesCopyright = null;
		Map<String, CopyrightLicense> licenses;
		
		
		// ROADMAP Implement more checks (field duplicates, etc.).
		
		filesCopyrights = new ArrayList<>();
		licenses = new LinkedHashMap<>();
		for (Field field : fields) {
			if (header) {
				// --- Ignore empty fields ---
				if (field.isEmpty()) {
					if (copyright != null) {
						header = false;
					}
				}
				// --- Report nameless fields ---
				else if (field.isNameless()) {
					if (copyright == null) {
						context.addWarning(new CopyrightUnsupportedFormatWarning());
						break;
					}
					throw new ParseException("Found nameless field |" + field.getValue() + "| in the header of the copyright.");
				} else {
					if (this.isField(field, FIELD_FORMAT)) {
						copyright = new CopyrightImpl();
						copyright.setFormat(field.getValue());
						
						// --- Leave the parsing if the format is not supported. ---
						if (!FORMAT_V1_0.equals(copyright.getFormat())) {
							
							context.addWarning(new CopyrightUnsupportedFormatWarning(copyright.getFormat()));
							break;
						}
					} else {
						// --- Leave the parsing if the first field isn't the format field ---
						if (copyright == null) {
							context.addWarning(new CopyrightUnsupportedFormatWarning());
							break;
						}
						
						if (this.isField(field, FIELD_UPSTREAM_NAME)) {
							copyright.setUpstreamName(field.getValue());
						} else if (this.isField(field, FIELD_UPSTREAM_CONTACT)) {
							copyright.setUpstreamContact(field.getValue());
						} else if (this.isField(field, FIELD_SOURCE)) {
							copyright.setSource(field.getValue());
						} else if (this.isField(field, FIELD_DISCLAIMER)) {
							copyright.setDisclaimer(field.getValue());
						} else if (this.isField(field, FIELD_COMMENT)) {
							copyright.setComment(field.getValue());
						} else if (this.isField(field, FIELD_LICENSE)) {
							license = this.parseLicense(field);
							copyright.setLicense(license);
							license = null;
						} else if (this.isField(field, FIELD_COPYRIGHT)) {
							copyright.setCopyright(field.getValue());
						} else {
							context.addWarning(new CopyrightUnsupportedFieldWarning(field.getName(), field.getValue()));
						}
					}
				}
			} else {
				if (copyright == null)
					throw new ParseException("Read empty line in copyright instead of header field.");
				
				// --- Ignore empty lines as separator ---
				if (field.isEmpty()) {
					filesCopyright = null;
					license = null;
				}
				// --- Report nameless fields ---
				else if (field.isNameless())
					throw new ParseException(
							"Found nameless field |" + field.getValue() + "| in the content of the copyright.");
				else if (this.isField(field, FIELD_FILES)) {
					if (license != null)
						throw new ParseException(
								"Read files copyright |" + field.getValue() + "| without read an empty line before it.");
					if (filesCopyright == null) {
						filesCopyright = this.parseFiles(field, null);
						filesCopyrights.add(filesCopyright);
					} else {
						this.parseFiles(field, filesCopyright);
					}
				}
				// --- Set the copyright in a copyright for files ---
				else if (this.isField(field, FIELD_COPYRIGHT)) {
					if (license != null)
						throw new ParseException("Read copyright field |" + field.getValue() + "| for license |"
								+ license.getName() + "|, but this field type is not supported for a license.");
					if (filesCopyright == null) {
						filesCopyright = new FilesCopyrightImpl();
						filesCopyrights.add(filesCopyright);
					}
					filesCopyright.setCopyright(field.getValue());
				}
				// --- Set the license in a copyright for files or add a global license ---
				else if (this.isField(field, FIELD_LICENSE)) {
					if (filesCopyright != null) {
						license = this.parseLicense(field);
						filesCopyright.setLicense(license);
						license = null;
					} else if (license != null)
						throw new ParseException(
								"Read license |" + field.getValue() + "| without read an empty line before it.");
					else {
						license = this.parseLicense(field);
						if (licenses.put(license.getName(), license) != null)
							throw new ParseException("Read license |" + license.getName() + "| more than one time.");
					}
				}
				// --- Set the comment in a copyright for files ---
				else if (this.isField(field, FIELD_COMMENT)) {
					if (license != null) {
						license.setComment(field.getValue());
					} else if (filesCopyright != null) {
						filesCopyright.setComment(field.getValue());
					} else
						throw new ParseException(
								"Found comment field |" + field.getValue() + "| outside of a license or files copyright.");
					
				}
			}
		}
		if (copyright == null) {
			copyright = new CopyrightImpl();
		} else {
			copyright.setFilesCopyrights(filesCopyrights);
			copyright.setLicenses(licenses);
		}
		return copyright;
	}


	/**
	 * <p>
	 * Parses a license from the specified field.
	 * </p>
	 *
	 * @param field
	 *            The field.
	 * @return The license.
	 */
	private CopyrightLicenseImpl parseLicense(Field field) {
		CopyrightLicenseImpl license;
		String value, line;
		String[] lines;
		StringBuilder sb;


		value = field.getFormattedValue();
		lines = value.split("\n", -1);
		license = new CopyrightLicenseImpl();
		license.setName(lines[0]);
		if (lines.length > 1) {
			sb = new StringBuilder();
			for (int index = 1; index < lines.length; index++) {
				line = lines[index];

				if (sb.length() > 0) {
					sb.append('\n');
				}
				sb.append(line);
			}
			license.setText(sb.toString());
		}
		return license;
	}


	/**
	 * <p>
	 * Parses the patterns from the specified field.
	 * </p>
	 *
	 * @param field
	 *            The field.
	 * @param filesCopyright
	 *            The copyright (optional).
	 * @return The copyright containing the patterns.
	 */
	private FilesCopyrightImpl parseFiles(Field field, FilesCopyrightImpl filesCopyright) {
		FilesCopyrightImpl fc;
		List<String> patterns;


		if (filesCopyright == null) {
			fc = new FilesCopyrightImpl();
		} else {
			fc = filesCopyright;
		}

		patterns = this.parsePatterns(field.getValue());
		fc.setFiles(patterns);
		return fc;
	}
	
	
	/**
	 * <p>
	 * Parses the patterns from the specified value.
	 * </p>
	 *
	 * @param value
	 *            The value.
	 * @return The patterns.
	 */
	private List<String> parsePatterns(String value) {
		String[] patterns;


		patterns = value.split("[\\n ]", -1);
		return Arrays.asList(patterns);
	}


	/**
	 * <p>
	 * Returns the flag if the field has the specified name.
	 * </p>
	 * <p>
	 * If the field doesn't have a name (means {@code null}), the method will
	 * return {@code false}.
	 * </p>
	 *
	 * @param field
	 *            The field.
	 * @param name
	 *            The name.
	 * @return The flag: {@code true}, if the field has the specified name,
	 *         {@code false} otherwise.
	 */
	private boolean isField(Field field, String name) {
		boolean match;

		if (field.getName() == null) {
			match = false;
		} else if (this.strictFieldNames) {
			match = name.equals(field.getName());
		} else {
			match = name.equalsIgnoreCase(field.getName());
		}
		return match;
	}


}
