/*
 * dpkg - Debian Package library and the Debian Package Maven plugin
 * (c) Copyright 2016 Gerrit Hohl
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package net.sourceforge.javadpkg.impl;

import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import net.sourceforge.javadpkg.Copyright;
import net.sourceforge.javadpkg.CopyrightLicense;
import net.sourceforge.javadpkg.FilesCopyright;


/**
 * <p>
 * A {@link Copyright} implementation.
 * </p>
 *
 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
 * @version <b>1.0</b>, 06.05.2016 by Gerrit Hohl
 */
public class CopyrightImpl implements Copyright {
	
	
	/** The format of the copyright. */
	private String							format;
	/** The name upstream uses for the software. */
	private String							upstreamName;
	/** The upstream contact. */
	private String							upstreamContact;
	/** The source. */
	private String							source;
	/** The disclaimer. */
	private String							disclaimer;
	/** The comment. */
	private String							comment;
	/** The license. */
	private CopyrightLicense				license;
	/** The copyright. */
	private String							copyright;
	/** The copyrights for certain files of a Debian package. */
	private List<FilesCopyright>			filesCopyrights;
	/** The licenses. */
	private Map<String, CopyrightLicense>	licenses;


	/**
	 * <p>
	 * Creates a copyright.
	 * </p>
	 */
	public CopyrightImpl() {
		super();
		
		this.format = null;
		this.upstreamName = null;
		this.upstreamContact = null;
		this.source = null;
		this.disclaimer = null;
		this.comment = null;
		this.license = null;
		this.copyright = null;
		this.filesCopyrights = new ArrayList<>();
		this.licenses = new LinkedHashMap<>();
	}
	
	
	@Override
	public String getFormat() {
		return this.format;
	}


	/**
	 * <p>
	 * Sets the format of the copyright.
	 * </p>
	 *
	 * @param format
	 *            The format.
	 */
	public void setFormat(String format) {
		this.format = format;
	}


	@Override
	public String getUpstreamName() {
		return this.upstreamName;
	}
	
	
	/**
	 * <p>
	 * Sets the name upstream uses for the software.
	 * </p>
	 *
	 * @param upstreamName
	 *            The name.
	 */
	public void setUpstreamName(String upstreamName) {
		this.upstreamName = upstreamName;
	}


	@Override
	public String getUpstreamContact() {
		return this.upstreamContact;
	}


	/**
	 * <p>
	 * Sets the upstream contact.
	 * </p>
	 *
	 * @param upstreamContact
	 *            The contact.
	 */
	public void setUpstreamContact(String upstreamContact) {
		this.upstreamContact = upstreamContact;
	}


	@Override
	public String getSource() {
		return this.source;
	}
	
	
	/**
	 * <p>
	 * Sets the source.
	 * </p>
	 *
	 * @param source
	 *            The source.
	 */
	public void setSource(String source) {
		this.source = source;
	}


	@Override
	public String getDisclaimer() {
		return this.disclaimer;
	}


	/**
	 * <p>
	 * Sets the disclaimer.
	 * </p>
	 *
	 * @param disclaimer
	 *            The disclaimer.
	 */
	public void setDisclaimer(String disclaimer) {
		this.disclaimer = disclaimer;
	}


	@Override
	public String getComment() {
		return this.comment;
	}


	/**
	 * <p>
	 * Sets the comment.
	 * </p>
	 *
	 * @param comment
	 *            The comment.
	 */
	public void setComment(String comment) {
		this.comment = comment;
	}


	@Override
	public CopyrightLicense getLicense() {
		return this.license;
	}
	
	
	/**
	 * <p>
	 * Sets the license.
	 * </p>
	 *
	 * @param license
	 *            The license.
	 */
	public void setLicense(CopyrightLicense license) {
		this.license = license;
	}


	@Override
	public String getCopyright() {
		return this.copyright;
	}
	
	
	/**
	 * <p>
	 * Sets the copyright.
	 * </p>
	 *
	 * @param copyright
	 *            The copyright.
	 */
	public void setCopyright(String copyright) {
		this.copyright = copyright;
	}


	@Override
	public List<FilesCopyright> getFilesCopyrights() {
		return (new ArrayList<>(this.filesCopyrights));
	}
	
	
	/**
	 * <p>
	 * Sets the copyrights for certain files of a Debian package.
	 * </p>
	 *
	 * @param filesCopyrights
	 *            The copyrights.
	 */
	public void setFilesCopyrights(List<FilesCopyright> filesCopyrights) {
		if (filesCopyrights == null) {
			this.filesCopyrights = new ArrayList<>();
		} else {
			this.filesCopyrights = new ArrayList<>(filesCopyrights);
		}
	}


	/**
	 * <p>
	 * Adds the copyright for certain files of a Debian package.
	 * </p>
	 *
	 * @param filesCopyright
	 *            The copyright.
	 * @throws IllegalArgumentException
	 *             If the copyright is <code>null</code>.
	 */
	public void addFilesCopyright(FilesCopyright filesCopyright) {
		if (filesCopyright == null)
			throw new IllegalArgumentException("Argument filesCopyright is null.");
		
		this.filesCopyrights.add(filesCopyright);
	}


	@Override
	public Map<String, CopyrightLicense> getLicenses() {
		return (new LinkedHashMap<>(this.licenses));
	}
	
	
	/**
	 * <p>
	 * Sets the licenses.
	 * </p>
	 *
	 * @param licenses
	 *            The licenses.
	 */
	public void setLicenses(Map<String, CopyrightLicense> licenses) {
		if (licenses == null) {
			this.licenses = new LinkedHashMap<>();
		} else {
			this.licenses = new LinkedHashMap<>(licenses);
		}
	}


	/**
	 * <p>
	 * Adds a license to the licenses.
	 * </p>
	 * <p>
	 * If a license with the same name already exists, then it will be replaced.
	 * </p>
	 *
	 * @param license
	 *            The license.
	 * @see #setLicenses(Map)
	 */
	public void addLicense(CopyrightLicense license) {
		if (license == null)
			throw new IllegalArgumentException("Argument license is null.");
		this.licenses.put(license.getName(), license);
	}


}
