/*
 * dpkg - Debian Package library and the Debian Package Maven plugin
 * (c) Copyright 2016 Gerrit Hohl
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package net.sourceforge.javadpkg.field.impl;

import java.io.BufferedWriter;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.util.List;

import net.sourceforge.javadpkg.BuildException;
import net.sourceforge.javadpkg.Context;
import net.sourceforge.javadpkg.GlobalConstants;
import net.sourceforge.javadpkg.field.Field;
import net.sourceforge.javadpkg.field.FieldBuilder;
import net.sourceforge.javadpkg.io.DataTarget;


/**
 * <p>
 * A {@link FieldBuilder} implementation.
 * </p>
 *
 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
 * @version <b>1.0</b>, 29.04.2016 by Gerrit Hohl
 */
public class FieldBuilderImpl implements FieldBuilder, GlobalConstants {
	
	
	/**
	 * <p>
	 * Creates a builder.
	 * </p>
	 */
	public FieldBuilderImpl() {
		super();
	}


	@Override
	public void buildFields(List<Field> fields, DataTarget target, Context context) throws IOException, BuildException {
		StringBuilder sb;
		String value;
		
		
		if (fields == null)
			throw new IllegalArgumentException("Argument fields is null.");
		if (target == null)
			throw new IllegalArgumentException("Argument target is null.");
		if (context == null)
			throw new IllegalArgumentException("Argument context is null.");
		
		sb = new StringBuilder();
		for (Field field : fields) {
			if (sb.length() > 0) {
				sb.append('\n');
			}
			
			if (field.isEmpty()) {
				continue;
			}
			
			if (!field.isNameless()) {
				sb.append(field.getName());
				sb.append(": ");
			}
			value = this.getFormattedValue(field.getValue());
			sb.append(value);
		}
		if (sb.length() > 0) {
			sb.append('\n');
		}

		try (BufferedWriter out = new BufferedWriter(new OutputStreamWriter(target.getOutputStream(), UTF_8_CHARSET))) {
			out.write(sb.toString());
		}
	}
	
	
	/**
	 * <p>
	 * Returns the formatted value.
	 * </p>
	 *
	 * @param value
	 *            The value.
	 * @return The formatted value.
	 */
	private String getFormattedValue(String value) {
		String[] values;
		StringBuilder sb;
		
		
		if (value.indexOf('\n') == -1)
			return value;
		
		// ROADMAP Throw a BuildException if a line is empty.
		values = value.split("\n", -1);
		sb = new StringBuilder();
		for (String v : values) {
			if (sb.length() > 0) {
				sb.append("\n ");
			}
			sb.append(v);
		}
		return sb.toString();
	}
	
	
}
