/*
 * dpkg - Debian Package library and the Debian Package Maven plugin
 * (c) Copyright 2016 Gerrit Hohl
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package net.sourceforge.javadpkg.field.impl;

import net.sourceforge.javadpkg.field.Field;

/**
 * <p>
 * An abstract {@link Field} implementation.
 * </p>
 *
 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
 * @version <b>1.0</b>, 06.05.2016 by Gerrit Hohl
 */
public abstract class AbstractField implements Field {
	
	
	/** The flag if the field is a nameless field. */
	private boolean	nameless;
	/** The flag if the field is an empty field. */
	private boolean	empty;


	/**
	 * <p>
	 * Creates a field.
	 * </p>
	 *
	 * @param nameless
	 *            The flag if the field is a nameless field.
	 * @param empty
	 *            The flag if the field is an empty field.
	 */
	public AbstractField(boolean nameless, boolean empty) {
		super();

		this.nameless = nameless;
		this.empty = empty;
	}


	/**
	 * <p>
	 * Formats the specified value.
	 * </p>
	 * <p>
	 * All empty lines are filled with a &quot;.&quot;.
	 * </p>
	 *
	 * @param value
	 *            The value.
	 * @return The formatted value.
	 */
	protected String formatValue(String value) {
		StringBuilder sb;
		String[] lines;


		if (value == null)
			return null;
		
		sb = new StringBuilder();
		lines = value.split("\n", -1);
		for (String line : lines) {
			if (sb.length() > 0) {
				sb.append('\n');
			}
			if (line.isEmpty()) {
				sb.append('.');
			} else {
				sb.append(line);
			}
		}
		return sb.toString();
	}
	
	
	@Override
	public boolean isNameless() {
		return this.nameless;
	}
	
	
	@Override
	public String getFormattedValue() {
		StringBuilder sb;
		String value;
		String[] lines;
		
		
		value = this.getValue();
		if (value == null)
			return null;
		
		sb = new StringBuilder();
		lines = value.split("\n", -1);
		for (String line : lines) {
			if (sb.length() > 0) {
				sb.append('\n');
			}
			if (!".".equals(line)) {
				sb.append(line);
			}
		}
		return sb.toString();
	}


	@Override
	public boolean isEmpty() {
		return this.empty;
	}


}
