/*
 * dpkg - Debian Package library and the Debian Package Maven plugin
 * (c) Copyright 2016 Gerrit Hohl
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package net.sourceforge.javadpkg.control.impl;

import java.util.Map;
import java.util.TreeMap;

import net.sourceforge.javadpkg.Context;
import net.sourceforge.javadpkg.ParseException;
import net.sourceforge.javadpkg.control.PackagePriority;
import net.sourceforge.javadpkg.control.PackagePriorityParser;


/**
 * <p>
 * A {@link PackagePriorityParser} implementation.
 * </p>
 *
 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
 * @version <b>1.0</b>, 01.01.2016 by Gerrit Hohl
 */
public class PackagePriorityParserImpl implements PackagePriorityParser {
	
	
	/** The priorities. */
	private Map<String, PackagePriority> priorities;
	
	
	/**
	 * <p>
	 * Creates a parser.
	 * </p>
	 */
	public PackagePriorityParserImpl() {
		super();
		
		this.priorities = new TreeMap<>();
		this.addPriority("required");
		this.addPriority("important");
		this.addPriority("standard");
		this.addPriority("optional");
		this.addPriority("extra");
	}


	/**
	 * <p>
	 * Adds a priority to the internal map.
	 * </p>
	 *
	 * @param text
	 *            The text.
	 */
	private void addPriority(String text) {
		this.priorities.put(text, new PackagePriorityImpl(text));
	}


	@Override
	public PackagePriority parsePackagePriority(String value, Context context) throws ParseException {
		PackagePriority priority;


		if (value == null)
			throw new IllegalArgumentException("Argument value is null.");
		if (context == null)
			throw new IllegalArgumentException("Argument context is null.");
		
		priority = this.priorities.get(value);
		if (priority == null) {
			context.addWarning(new PackagePriorityUnsupportedWarning(value));
			priority = new PackagePriorityImpl(value);
		}
		
		return priority;
	}


	/* **********************************************************************
	 * **********************************************************************
	 * **********************************************************************
	 * **********************************************************************
	 * **********************************************************************
	 */


	/**
	 * <p>
	 * The {@link PackagePriority} implementation of this class.
	 * </p>
	 *
	 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
	 * @version <b>1.0</b>, 01.01.2016 by Gerrit Hohl
	 */
	private class PackagePriorityImpl implements PackagePriority {
		
		
		/** The text. */
		private String text;
		
		
		/**
		 * <p>
		 * Creates a priority.
		 * </p>
		 *
		 * @param text
		 *            The text.
		 */
		public PackagePriorityImpl(String text) {
			super();

			this.text = text;
		}
		
		
		@Override
		public String getText() {
			return this.text;
		}


	}


}
