/*
 * dpkg - Debian Package library and the Debian Package Maven plugin
 * (c) Copyright 2016 Gerrit Hohl
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package net.sourceforge.javadpkg.control.impl;

import java.util.Map;
import java.util.TreeMap;

import net.sourceforge.javadpkg.Context;
import net.sourceforge.javadpkg.ParseException;
import net.sourceforge.javadpkg.control.PackageMultiArchitecture;
import net.sourceforge.javadpkg.control.PackageMultiArchitectureParser;


/**
 * <p>
 * A {@link PackageMultiArchitectureParser} implementation.
 * </p>
 *
 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
 * @version <b>1.0</b>, 03.01.2016 by Gerrit Hohl
 */
public class PackageMultiArchitectureParserImpl implements PackageMultiArchitectureParser {
	
	
	/** The properties. */
	private Map<String, PackageMultiArchitecture> multiArchitectures;


	/**
	 * <p>
	 * Creates a parser.
	 * </p>
	 */
	public PackageMultiArchitectureParserImpl() {
		super();
		
		this.multiArchitectures = new TreeMap<>();
		this.addMultiArchitecture("same");
		this.addMultiArchitecture("foreign");
		this.addMultiArchitecture("allowed");
	}
	
	
	/**
	 * <p>
	 * Adds a property.
	 * </p>
	 *
	 * @param multiArchitecture
	 *            The property.
	 */
	private void addMultiArchitecture(String multiArchitecture) {
		this.multiArchitectures.put(multiArchitecture, new PackageMultiArchitectureImpl(multiArchitecture));
	}
	
	
	@Override
	public PackageMultiArchitecture parsePackageMultiArchitecture(String value, Context context) throws ParseException {
		PackageMultiArchitecture multiArchitecture;
		
		
		if (value == null)
			throw new IllegalArgumentException("Argument value is null.");
		if (context == null)
			throw new IllegalArgumentException("Argument context is null.");
		
		multiArchitecture = this.multiArchitectures.get(value);
		if (multiArchitecture == null) {
			context.addWarning(new PackageMultiArchitectureUnsupportedWarning(value));
			multiArchitecture = new PackageMultiArchitectureImpl(value);
		}

		return multiArchitecture;
	}


	/* **********************************************************************
	 * **********************************************************************
	 * **********************************************************************
	 * **********************************************************************
	 * **********************************************************************
	 */


	/**
	 * <p>
	 * The {@link PackageMultiArchitecture} implementation of this class.
	 * </p>
	 *
	 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
	 * @version <b>1.0</b>, 03.01.2016 by Gerrit Hohl
	 */
	private class PackageMultiArchitectureImpl implements PackageMultiArchitecture {
		
		
		/** The text. */
		private String text;


		/**
		 * <p>
		 * Creates a multiple architecture property.
		 * </p>
		 *
		 * @param text
		 *            The text.
		 */
		public PackageMultiArchitectureImpl(String text) {
			super();

			this.text = text;
		}


		@Override
		public String getText() {
			return this.text;
		}


	}
	
	
}
