/*
 * dpkg - Debian Package library and the Debian Package Maven plugin
 * (c) Copyright 2016 Gerrit Hohl
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package net.sourceforge.javadpkg.control.impl;

import java.util.ArrayList;
import java.util.List;

import net.sourceforge.javadpkg.Context;
import net.sourceforge.javadpkg.ParseException;
import net.sourceforge.javadpkg.control.PackageDependency;
import net.sourceforge.javadpkg.control.PackageDependency.Condition;
import net.sourceforge.javadpkg.control.PackageDependencyParser;
import net.sourceforge.javadpkg.control.PackageName;
import net.sourceforge.javadpkg.control.PackageNameParser;
import net.sourceforge.javadpkg.control.PackageVersion;
import net.sourceforge.javadpkg.control.PackageVersionParser;
import net.sourceforge.javadpkg.control.PackageVersionRelationOperator;
import net.sourceforge.javadpkg.control.PackageVersionRelationOperatorParser;


/**
 * <p>
 * A {@link PackageDependencyParser} implementation.
 * </p>
 *
 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
 * @version <b>1.0</b>, 01.01.2016 by Gerrit Hohl
 * @version <b>1.1</b>, 23.01.2018 by Gerrit Hohl
 */
public class PackageDependencyParserImpl implements PackageDependencyParser {


	/** The parser for the package names. */
	private PackageNameParser						packageNameParser;
	/** The parser for the relation operator. */
	private PackageVersionRelationOperatorParser	packageVersionRelationOperatorParser;
	/** The parser for the package version. */
	private PackageVersionParser					packageVersionParser;


	/**
	 * <p>
	 * Creates a parser.
	 * </p>
	 *
	 * @param packageNameParser
	 *            The parser for the package names.
	 * @param packageVersionRelationOperatorParser
	 *            The parser for the relation operator.
	 * @param packageVersionParser
	 *            The parser for the package version.
	 * @throws IllegalArgumentException
	 *             If any of the parameters are <code>null</code>.
	 */
	public PackageDependencyParserImpl(PackageNameParser packageNameParser,
			PackageVersionRelationOperatorParser packageVersionRelationOperatorParser,
			PackageVersionParser packageVersionParser) {

		super();

		if (packageNameParser == null)
			throw new IllegalArgumentException("Argument packageNameParser is null.");
		if (packageVersionRelationOperatorParser == null)
			throw new IllegalArgumentException("Argument packageVersionRelationOperatorParser is null.");
		if (packageVersionParser == null)
			throw new IllegalArgumentException("Argument packageVersionParser is null.");

		this.packageNameParser = packageNameParser;
		this.packageVersionRelationOperatorParser = packageVersionRelationOperatorParser;
		this.packageVersionParser = packageVersionParser;
	}


	@Override
	public PackageDependency parsePackageDependency(String value, Context context) throws ParseException {
		PackageDependency dependency;
		List<PackageDependency> dependencies;
		String[] parts;
		PackageName packageName;
		PackageVersionRelationOperator relationOperator = null;
		PackageVersion version = null;
		String part;
		
		
		if (value == null)
			throw new IllegalArgumentException("Argument value is null.");
		if (context == null)
			throw new IllegalArgumentException("Argument context is null.");

		// --- Does the value contain an OR condition? ---
		parts = value.split("\\|");
		if (parts.length > 1) {
			dependencies = new ArrayList<>();
			for (String p : parts) {
				try {
					dependency = this.parsePackageDependency(p.trim(), context);
				} catch (ParseException e) {
					throw new ParseException("Couldn't parse dependencies |" + value + "|: " + e.getMessage(), e);
				}
				dependencies.add(dependency);
			}
			dependency = new PackageDependencyImpl(Condition.OR, dependencies);
		}
		// --- Otherwise it is a normal value ---
		else {
			parts = value.split("[ \\t]+", 2);
			try {
				// --- Get name ---
				packageName = this.packageNameParser.parsePackageName(parts[0], context);
				// --- Do we also have a version and maybe a relational operator? ---
				if (parts.length > 1) {
					// --- Get parts ---
					part = parts[1];
					if (part.startsWith("(") && part.endsWith(")")) {
						part = part.substring(1, part.length() - 1);
						parts = part.split(" ", 2);
						// --- Maybe there is no whitespace ---
						if (parts.length == 1) {
							// --- Then we split the part before the 1st number from the part starting at that number ---
							parts = part.split("([0-9]){1}", 2);
							if ((parts.length == 1) || (parts[0].length() == 0)) {
								parts = new String[] { part };
							} else {
								parts[1] = part.substring(parts[0].length());
							}
						}
					} else
						throw new ParseException("Expected a version criteria, but found |" + part + "|.");

					// --- Do we have a relational operator? ---
					if (parts.length > 1) {
						relationOperator = this.packageVersionRelationOperatorParser
								.parsePackageVersionRelationOperator(parts[0]);
						version = this.packageVersionParser.parsePackageVersion(parts[1], context);
					}
					// --- Otherwise parse only the version ---
					else {
						version = this.packageVersionParser.parsePackageVersion(parts[0], context);
					}
				}
			} catch (ParseException e) {
				throw new ParseException("Couldn't parse package dependency |" + value + "|: " + e.getMessage(), e);
			}
			dependency = new PackageDependencyImpl(packageName, relationOperator, version);
		}
		return dependency;
	}
	
	
	@Override
	public List<PackageDependency> parsePackageDependencies(String values, Context context) throws ParseException {
		List<PackageDependency> dependencies;
		String[] parts;
		PackageDependency dependency;
		
		
		if (values == null)
			throw new IllegalArgumentException("Argument values is null.");
		if (context == null)
			throw new IllegalArgumentException("Argument context is null.");

		dependencies = new ArrayList<>();
		parts = values.split(",");
		for (String part : parts) {
			try {
				dependency = this.parsePackageDependency(part.trim(), context);
			} catch (ParseException e) {
				throw new ParseException("Couldn't parse dependencies |" + values + "|: " + e.getMessage(), e);
			}
			dependencies.add(dependency);
		}
		return dependencies;
	}
	
	
	/* **********************************************************************
	 * **********************************************************************
	 * **********************************************************************
	 * **********************************************************************
	 * **********************************************************************
	 */
	
	
	/**
	 * <p>
	 * The {@link PackageDependency} implementation of this class.
	 * </p>
	 *
	 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
	 * @version <b>1.0</b>, 01.01.2016 by Gerrit Hohl
	 */
	private class PackageDependencyImpl implements PackageDependency {


		/** The condition. */
		private Condition						condition;
		/** The package dependencies for the condition. */
		private List<PackageDependency>			packageDependencies;
		/** The name of the package. */
		private PackageName						packageName;
		/** The relational operator (optional). */
		private PackageVersionRelationOperator	relationOperator;
		/** The version (optional). */
		private PackageVersion					version;


		/**
		 * <p>
		 * Creates a package dependency.
		 * </p>
		 *
		 * @param condition
		 *            The condition.
		 * @param packageDependencies
		 *            The package dependencies for the condition.
		 */
		public PackageDependencyImpl(Condition condition, List<PackageDependency> packageDependencies) {
			super();
			
			this.condition = condition;
			this.packageDependencies = new ArrayList<>(packageDependencies);
			this.packageName = null;
			this.relationOperator = null;
			this.version = null;
		}


		/**
		 * <p>
		 * Creates a package dependency.
		 * </p>
		 *
		 * @param packageName
		 *            The name of the package.
		 * @param relationOperator
		 *            The relational operator (optional).
		 * @param version
		 *            The version (optional).
		 */
		public PackageDependencyImpl(PackageName packageName, PackageVersionRelationOperator relationOperator,
				PackageVersion version) {

			super();
			
			this.condition = null;
			this.packageDependencies = new ArrayList<>();
			this.packageName = packageName;
			this.relationOperator = relationOperator;
			this.version = version;
		}


		@Override
		public boolean isCondition() {
			return (this.condition != null);
		}


		@Override
		public Condition getCondition() {
			return this.condition;
		}


		@Override
		public List<PackageDependency> getConditionPackageDependencies() {
			return (new ArrayList<>(this.packageDependencies));
		}


		@Override
		public PackageName getPackage() {
			return this.packageName;
		}


		@Override
		public PackageVersionRelationOperator getRelationOperator() {
			return this.relationOperator;
		}


		@Override
		public PackageVersion getVersion() {
			return this.version;
		}


	}


}
