/*
 * dpkg - Debian Package library and the Debian Package Maven plugin
 * (c) Copyright 2016 Gerrit Hohl
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package net.sourceforge.javadpkg.control.impl;

import java.util.List;

import net.sourceforge.javadpkg.BuildException;
import net.sourceforge.javadpkg.Context;
import net.sourceforge.javadpkg.control.PackageDependency;
import net.sourceforge.javadpkg.control.PackageDependencyBuilder;
import net.sourceforge.javadpkg.control.PackageNameBuilder;
import net.sourceforge.javadpkg.control.PackageVersionBuilder;
import net.sourceforge.javadpkg.control.PackageVersionRelationOperatorBuilder;


/**
 * <p>
 * A {@link PackageDependencyBuilder} implementation.
 * </p>
 *
 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
 * @version <b>1.0</b>, 28.04.2016 by Gerrit Hohl
 */
public class PackageDependencyBuilderImpl implements PackageDependencyBuilder {


	/** The builder for the package names. */
	private PackageNameBuilder						packageNameBuilder;
	/** The builder for the relation operator. */
	private PackageVersionRelationOperatorBuilder	packageVersionRelationOperatorBuilder;
	/** The builder for the package version. */
	private PackageVersionBuilder					packageVersionBuilder;


	/**
	 * <p>
	 * Creates a builder.
	 * </p>
	 *
	 * @param packageNameBuilder
	 *            The builder for the package names.
	 * @param packageVersionRelationOperatorBuilder
	 *            The builder for the relation operator.
	 * @param packageVersionBuilder
	 *            The builder for the package version.
	 * @throws IllegalArgumentException
	 *             If any of the parameters are <code>null</code>.
	 */
	public PackageDependencyBuilderImpl(PackageNameBuilder packageNameBuilder,
			PackageVersionRelationOperatorBuilder packageVersionRelationOperatorBuilder,
			PackageVersionBuilder packageVersionBuilder) {
			
		super();

		if (packageNameBuilder == null)
			throw new IllegalArgumentException("Argument packageNameBuilder is null.");
		if (packageVersionRelationOperatorBuilder == null)
			throw new IllegalArgumentException("Argument packageVersionRelationOperatorBuilder is null.");
		if (packageVersionBuilder == null)
			throw new IllegalArgumentException("Argument packageVersionBuilder is null.");
			
		this.packageNameBuilder = packageNameBuilder;
		this.packageVersionRelationOperatorBuilder = packageVersionRelationOperatorBuilder;
		this.packageVersionBuilder = packageVersionBuilder;
	}


	@Override
	public String buildPackageDependency(PackageDependency dependency, Context context) throws BuildException {
		String value;
		StringBuilder sb;
		
		
		if (dependency == null)
			throw new IllegalArgumentException("Argument dependency is null.");
		if (context == null)
			throw new IllegalArgumentException("Argument context is null.");
			
		// --- Do we have a condition? ---
		if (dependency.isCondition()) {
			sb = new StringBuilder();
			// --- Add all dependencies including the condition ---
			for (PackageDependency dep : dependency.getConditionPackageDependencies()) {
				if (sb.length() > 0) {
					switch (dependency.getCondition()) {
						case OR:
							sb.append(" | ");
							break;
							
						default:
							throw new BuildException("Unsupported dependency condition |" + dependency.getCondition() + "|.");
					}
					
				}
				value = this.buildPackageDependency(dep, context);
				sb.append(value);
			}
			value = sb.toString();
		}
		// --- Otherwise we have a simple dependency ---
		else {
			sb = new StringBuilder();

			// --- Package name ---
			try {
				value = this.packageNameBuilder.buildPackageName(dependency.getPackage(), context);
			} catch (BuildException e) {
				throw new BuildException("Couldn't build dependency: " + e.getMessage(), e);
			}
			sb.append(value);
			
			if (dependency.getVersion() != null) {
				sb.append(" (");
				// --- Relational operator ---
				if (dependency.getRelationOperator() != null) {
					try {
						value = this.packageVersionRelationOperatorBuilder
								.buildPackageVersionRelationOperator(dependency.getRelationOperator(), context);
					} catch (BuildException e) {
						throw new BuildException("Couldn't build dependency: " + e.getMessage(), e);
					}
					sb.append(value);
					sb.append(' ');
				}

				// --- Version ---
				try {
					value = this.packageVersionBuilder.buildPackageVersion(dependency.getVersion(), context);
				} catch (BuildException e) {
					throw new BuildException("Couldn't build dependency: " + e.getMessage(), e);
				}
				sb.append(value);
				sb.append(')');
			}
			value = sb.toString();
		}
		return value;
	}


	@Override
	public String buildPackageDependencies(List<PackageDependency> dependencies, Context context) throws BuildException {
		StringBuilder result;
		String value;
		
		
		if (dependencies == null)
			throw new IllegalArgumentException("Argument dependencies is null.");
		if (context == null)
			throw new IllegalArgumentException("Argument context is null.");

		result = new StringBuilder();
		for (PackageDependency dependency : dependencies) {
			if (result.length() > 0) {
				result.append(", ");
			}
			
			try {
				value = this.buildPackageDependency(dependency, context);
			} catch (BuildException e) {
				throw new BuildException("Couldn't build dependencies: " + e.getMessage(), e);
			}
			result.append(value);
		}
		return result.toString();
	}


}
