/*
 * dpkg - Debian Package library and the Debian Package Maven plugin
 * (c) Copyright 2016 Gerrit Hohl
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package net.sourceforge.javadpkg;


/**
 * <p>
 * The paths of the files in the document folder of the Debian package.
 * </p>
 *
 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
 * @version <b>1.0</b>, 06.05.2016 by Gerrit Hohl
 */
public interface DocumentPaths {
	
	
	/**
	 * <p>
	 * Returns the base path of the document folder.
	 * </p>
	 * <p>
	 * Normally this should be: <blockquote>
	 * <code>/usr/share/doc/</code></blockquote>
	 * </p>
	 *
	 * @return The base path.
	 */
	String getDocumentBasePath();
	
	
	/**
	 * <p>
	 * Returns the path of the document folder of the Debian package.
	 * </p>
	 * <p>
	 * Normally this should be: <blockquote>
	 * <code>/usr/share/doc/[package-name]/</code></blockquote>
	 * </p>
	 *
	 * @return The path.
	 */
	String getDocumentPath();


	/**
	 * <p>
	 * Returns the flag if the specified path is the path of the copyright file.
	 * </p>
	 *
	 * @param path
	 *            The path.
	 * @return The flag: <code>true</code>, if the path is the path of the
	 *         copyright file, <code>false</code> otherwise.
	 * @throws IllegalArgumentException
	 *             If the path is <code>null</code>.
	 */
	boolean isCopyrightPath(String path);
	
	
	/**
	 * <p>
	 * Returns the path of the copyright file.
	 * </p>
	 * <p>
	 * Normally this should be: <blockquote>
	 * <code>/usr/share/doc/[package-name]/copyright</code></blockquote>
	 * </p>
	 *
	 * @return The path.
	 */
	String getCopyrightPath();
	
	
	/**
	 * <p>
	 * Returns the flag if the specified path is the path of the change log
	 * file.
	 * </p>
	 * <p>
	 * The method doesn't determine the format and/or compression of the change
	 * log file.
	 * </p>
	 *
	 * @param path
	 *            The path.
	 * @return The flag: <code>true</code>, if the path is the path of the
	 *         change log file, <code>false</code> otherwise.
	 * @throws IllegalArgumentException
	 *             If the path is <code>null</code>.
	 * @see #isChangeLogGzipPath(String)
	 * @see #isChangeLogHtmlPath(String)
	 */
	boolean isChangeLogPath(String path);
	
	
	/**
	 * <p>
	 * Returns the flag if the specified path is the path of the Debian change
	 * log file.
	 * </p>
	 * <pThe method doesn't determine the format and/or compression of the
	 * change log file.
	 * </p>
	 * <p>
	 * The filename of the Debian change log starts with
	 * &quot;changelog.Debian.*&quot;. It often exists parallel to the
	 * &quot;changelog.*&quot; file which in this case has its own, non-standard
	 * format. If both files exists the Debian change log should always be used.
	 * </p>
	 *
	 * @param path
	 *            The path.
	 * @return The flag: <code>true</code>, if the path is the path of the
	 *         Debian change log file, <code>false</code> otherwise.
	 * @throws IllegalArgumentException
	 *             If the path is <code>null</code>.
	 */
	boolean isChangeLogDebianPath(String path);
	
	
	/**
	 * <p>
	 * Returns the flag if the specified path is the path of a GZIP compressed
	 * change log file.
	 * </p>
	 * <p>
	 * The method doesn't determine the format of the change log file.
	 * </p>
	 *
	 * @param path
	 *            The path.
	 * @return The flag: <code>true</code>, if the path is the path of a GZIP
	 *         compressed change log file, <code>false</code> otherwise.
	 * @see #isChangeLogHtmlPath(String)
	 */
	boolean isChangeLogGzipPath(String path);
	
	
	/**
	 * <p>
	 * Returns the flag if the specified path is the path of a HTML formatted
	 * change log file.
	 * </p>
	 * <p>
	 * The method doesn't determine the compression of the change log file.
	 * </p>
	 *
	 * @param path
	 *            The path.
	 * @return The flag: <code>true</code>, if the path is the path of a HTML
	 *         formatted change log file, <code>false</code> otherwise.
	 * @see #isChangeLogGzipPath(String)
	 */
	boolean isChangeLogHtmlPath(String path);
	
	
	/**
	 * <p>
	 * Returns the path of the change log file.
	 * </p>
	 * <p>
	 * Normally this should be: <blockquote>
	 * <code>/usr/share/doc/[package-name]/changelog</code></blockquote>
	 * </p>
	 *
	 * @return The path.
	 */
	String getChangeLogPath();
	
	
	/**
	 * <p>
	 * Returns the path of the GZIP compressed change log file.
	 * </p>
	 * <p>
	 * Normally this should be: <blockquote>
	 * <code>/usr/share/doc/[package-name]/changelog.gz</code></blockquote>
	 * </p>
	 *
	 * @return The path.
	 */
	String getChangeLogGzipPath();
	
	
	/**
	 * <p>
	 * Returns the path of the HTML formatted change log file.
	 * </p>
	 * <p>
	 * Normally this should be: <blockquote>
	 * <code>/usr/share/doc/[package-name]/changelog.html</code></blockquote>
	 * </p>
	 *
	 * @return The path.
	 */
	String getChangeLogHtmlPath();
	
	
	/**
	 * <p>
	 * Returns the path of the GZIP compressed HTML formatted change log file.
	 * </p>
	 * <p>
	 * Normally this should be: <blockquote>
	 * <code>/usr/share/doc/[package-name]/changelog.html.gz</code></blockquote>
	 * </p>
	 *
	 * @return The path.
	 */
	String getChangeLogHtmlGzipPath();
	
	
}
