/*
 * dpkg - Debian Package library and the Debian Package Maven plugin
 * (c) Copyright 2015 Gerrit Hohl
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package net.sourceforge.javadpkg;

import net.sourceforge.javadpkg.control.Control;

/**
 * <p>
 * A Debian package.
 * </p>
 * <p>
 * See <a href="https://www.debian.org/doc/debian-policy/ch-controlfields.html">
 * Chapter 5 - Control files and their fields</a> for further information.
 * </p>
 *
 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
 * @version <b>1.0</b>, 27.12.2015 by Gerrit Hohl
 */
public interface DebianPackage {
	
	
	// TODO Add the format of the archive for the data.
	
	
	/**
	 * <p>
	 * Returns the version of the file format.
	 * </p>
	 * <p>
	 * In most cases this should be &quot;2.0&quot;
	 * </p>
	 *
	 * @return The version.
	 */
	String getFileFormatVersion();
	
	
	/**
	 * <p>
	 * Returns the control of the package.
	 * </p>
	 *
	 * @return The control.
	 */
	Control getControl();
	
	
	/**
	 * <p>
	 * Returns the MD5 sums.
	 * </p>
	 *
	 * @return The MD5 sums or <code>null</code>, if the Debian package doesn't
	 *         contain the MD5 sums.
	 */
	MD5Sums getMD5Sums();


	/**
	 * <p>
	 * Returns the script which will be executed before the package is
	 * installed.
	 * </p>
	 *
	 * @return The script or <code>null</code>, if the script doesn't exist.
	 */
	Script getPreInstall();


	/**
	 * <p>
	 * Returns the script which will be executed after the package is installed.
	 * </p>
	 *
	 * @return The script or <code>null</code>, if the script doesn't exist.
	 */
	Script getPostInstall();


	/**
	 * <p>
	 * Returns the script which will be executed before the package is removed.
	 * </p>
	 *
	 * @return The script or <code>null</code>, if the script doesn't exist.
	 */
	Script getPreRemove();


	/**
	 * <p>
	 * Returns the script which will be executed after the package is removed.
	 * </p>
	 *
	 * @return The script or <code>null</code>, if the script doesn't exist.
	 */
	Script getPostRemove();
	
	
	/**
	 * <p>
	 * Returns the templates for asking the user questions about the
	 * configuration.
	 * </p>
	 *
	 * @return The templates or <code>null</code>, if no templates exist.
	 * @see #getConfig()
	 */
	Templates getTemplates();


	/**
	 * <p>
	 * Returns the script for the configuration of the package.
	 * </p>
	 *
	 * @return The script or <code>null</code>, if the script doesn't exist.
	 * @see #getTemplates()
	 */
	Script getConfig();


	/**
	 * <p>
	 * Returns the configuration files which are managed by this package.
	 * </p>
	 *
	 * @return The configuration files or <code>null</code>, if no configuration
	 *         files are specified.
	 */
	ConfigFiles getConfigFiles();
	
	
	/**
	 * <p>
	 * Returns the shared libraries of this package.
	 * </p>
	 *
	 * @return The shared libraries or <code>null</code>, if no shared libraries
	 *         are specified.
	 */
	SharedLibraries getSharedLibraries();


	/**
	 * <p>
	 * Returns the symbols of the shared libraries of this package.
	 * </p>
	 *
	 * @return The symbols or <code>null</code>, if no symbols are specified.
	 */
	Symbols getSymbols();


	/**
	 * <p>
	 * Returns the copyright of this package.
	 * </p>
	 * 
	 * @return The copyright or <code>null</code>, if no copyright is specified.
	 */
	Copyright getCopyright();


	/**
	 * <p>
	 * Returns the change log of this package.
	 * </p>
	 * 
	 * @return The change log or <code>null</code>, if no change log is
	 *         specified.
	 */
	ChangeLog getChangeLog();
	
	
}
