/*
 * dpkg - Debian Package library and the Debian Package Maven plugin
 * (c) Copyright 2016 Gerrit Hohl
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package net.sourceforge.javadpkg.plugin.io.impl;

import org.junit.Assert;
import org.junit.Test;

import net.sourceforge.javadpkg.plugin.AbstractDpkgPluginTest;
import net.sourceforge.javadpkg.plugin.io.Path;
import net.sourceforge.javadpkg.plugin.io.impl.PathImpl;


/**
 * <p>
 * Performs some tests on the {@link PathImpl} class.
 * </p>
 *
 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
 * @version <b>1.0</b>, 10.05.2016 by Gerrit Hohl
 */
public class PathImplTest extends AbstractDpkgPluginTest {
	
	
	/**
	 * <p>
	 * Performs some tests on the {@link PathImpl#PathImpl()} constructor.
	 * </p>
	 */
	@Test
	public void testPathImpl() {
		Path path;


		// --- Root ---
		path = new PathImpl();
		Assert.assertEquals("/", path.getAbsolutePath());
		Assert.assertNull(path.getFirstElement());
		Assert.assertNull(path.getLastElement());
		Assert.assertNull(path.getParentPath());
		Assert.assertNull(path.getChildPath());
	}


	/**
	 * <p>
	 * Performs some tests on the {@link PathImpl#parsePath(String)} method.
	 * </p>
	 */
	@Test
	public void testParsePath() {
		Path path, parent, child;


		// --- Parameters ---
		try {
			PathImpl.parsePath(null);
			Assert.fail("Expected an exception, but none was thrown.");
			return;
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		}


		// --- Root node ---
		path = PathImpl.parsePath("");
		Assert.assertNotNull(path);
		Assert.assertEquals("/", path.getAbsolutePath());
		Assert.assertNull(path.getFirstElement());
		Assert.assertNull(path.getLastElement());
		Assert.assertNull(path.getParentPath());
		Assert.assertNull(path.getChildPath());


		// --- Sub-directory ---
		path = PathImpl.parsePath("/usr/local");
		Assert.assertEquals("usr", path.getFirstElement());
		Assert.assertEquals("local", path.getLastElement());
		Assert.assertNotNull(path);
		Assert.assertEquals("/usr/local", path.getAbsolutePath());
		parent = path.getParentPath();
		Assert.assertNotNull(parent);
		Assert.assertEquals("/usr", parent.getAbsolutePath());
		Assert.assertEquals("usr", parent.getFirstElement());
		Assert.assertEquals("usr", parent.getLastElement());
		parent = parent.getParentPath();
		Assert.assertNotNull(parent);
		Assert.assertEquals("/", parent.getAbsolutePath());
		Assert.assertNull(parent.getFirstElement());
		Assert.assertNull(parent.getLastElement());
		parent = parent.getParentPath();
		Assert.assertNull(parent);
		child = path.getChildPath();
		Assert.assertNotNull(child);
		Assert.assertEquals("/local", child.getAbsolutePath());
		Assert.assertEquals("local", child.getFirstElement());
		Assert.assertEquals("local", child.getLastElement());
		child = child.getChildPath();
		Assert.assertNull(child);


		// --- Sub-directory ---
		path = PathImpl.parsePath("/usr/local/");
		Assert.assertEquals("usr", path.getFirstElement());
		Assert.assertEquals("local", path.getLastElement());
		Assert.assertNotNull(path);
		Assert.assertEquals("/usr/local", path.getAbsolutePath());
		parent = path.getParentPath();
		Assert.assertNotNull(parent);
		Assert.assertEquals("/usr", parent.getAbsolutePath());
		Assert.assertEquals("usr", parent.getFirstElement());
		Assert.assertEquals("usr", parent.getLastElement());
		parent = parent.getParentPath();
		Assert.assertNotNull(parent);
		Assert.assertEquals("/", parent.getAbsolutePath());
		Assert.assertNull(parent.getFirstElement());
		Assert.assertNull(parent.getLastElement());
		parent = parent.getParentPath();
		Assert.assertNull(parent);
		child = path.getChildPath();
		Assert.assertNotNull(child);
		Assert.assertEquals("/local", child.getAbsolutePath());
		Assert.assertEquals("local", child.getFirstElement());
		Assert.assertEquals("local", child.getLastElement());
		child = child.getChildPath();
		Assert.assertNull(child);
	}


	/**
	 * <p>
	 * Performs some tests on the {@link PathImpl#createChild(String)} method.
	 * </p>
	 */
	@Test
	public void testAddChild() {
		Path path, child;


		// --- Add child ---
		path = PathImpl.parsePath("/usr/local");
		child = path.createChild("myprogram");
		Assert.assertEquals("/usr/local", path.getAbsolutePath());
		Assert.assertEquals("usr", path.getFirstElement());
		Assert.assertEquals("local", path.getLastElement());
		Assert.assertNotNull(child);
		Assert.assertEquals("/usr/local/myprogram", child.getAbsolutePath());
		Assert.assertEquals("usr", child.getFirstElement());
		Assert.assertEquals("myprogram", child.getLastElement());
	}
	
	
}
