/*
 * dpkg - Debian Package library and the Debian Package Maven plugin
 * (c) Copyright 2016 Gerrit Hohl
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package net.sourceforge.javadpkg.plugin.io.impl;

import java.io.File;

import org.junit.Assert;
import org.junit.Test;

import net.sourceforge.javadpkg.plugin.AbstractDpkgPluginTest;
import net.sourceforge.javadpkg.plugin.io.FileSystemNode;
import net.sourceforge.javadpkg.plugin.io.Path;
import net.sourceforge.javadpkg.plugin.io.impl.FileSystemNodeImpl;
import net.sourceforge.javadpkg.plugin.io.impl.PathImpl;


/**
 * <p>
 * Performs some tests on the {@link FileSystemNodeImpl} class.
 * </p>
 *
 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
 * @version <b>1.0</b>, 10.05.2016 by Gerrit Hohl
 */
public class FileSystemNodeImplTest extends AbstractDpkgPluginTest {
	
	
	/**
	 * <p>
	 * Performs some tests on the
	 * {@link FileSystemNodeImpl#FileSystemNodeImpl(String)} constructor.
	 * </p>
	 */
	@Test
	public void testFileSystemNodeImplString() {
		FileSystemNode<File> root, usr, local;


		// --- Root node ---
		root = new FileSystemNodeImpl<>("");
		Assert.assertNull(root.getParent());
		Assert.assertEquals("", root.getName());
		Assert.assertEquals("/", root.getPath().getAbsolutePath());
		Assert.assertTrue(root.isDirectory());
		Assert.assertFalse(root.isFile());
		Assert.assertTrue(root.isCreatedByDependency());
		Assert.assertTrue(root.containsOnlyCreatedByDependency());

		// --- usr node ---
		usr = new FileSystemNodeImpl<>("usr");
		root.addChild(usr);
		Assert.assertSame(root, usr.getParent());
		Assert.assertEquals("usr", usr.getName());
		Assert.assertEquals("/usr", usr.getPath().getAbsolutePath());
		Assert.assertTrue(usr.isDirectory());
		Assert.assertFalse(usr.isFile());
		Assert.assertTrue(usr.isCreatedByDependency());
		Assert.assertTrue(root.containsOnlyCreatedByDependency());
		
		// --- local node ---
		local = new FileSystemNodeImpl<>("local");
		usr.addChild(local);
		Assert.assertSame(usr, local.getParent());
		Assert.assertEquals("local", local.getName());
		Assert.assertEquals("/usr/local", local.getPath().getAbsolutePath());
		Assert.assertTrue(local.isDirectory());
		Assert.assertFalse(local.isFile());
		Assert.assertTrue(local.isCreatedByDependency());
		Assert.assertTrue(root.containsOnlyCreatedByDependency());
	}
	
	
	/**
	 * <p>
	 * Performs some tests on the
	 * {@link FileSystemNodeImpl#createDirectories(Path)} constructor.
	 * </p>
	 */
	@Test
	public void testCreateDirectories() {
		FileSystemNode<File> root, node;
		Path path;
		
		
		root = new FileSystemNodeImpl<>("");
		Assert.assertNull(root.getParent());
		Assert.assertEquals("", root.getName());
		Assert.assertEquals("/", root.getPath().getAbsolutePath());
		Assert.assertTrue(root.isDirectory());
		Assert.assertFalse(root.isFile());
		Assert.assertTrue(root.isCreatedByDependency());
		Assert.assertTrue(root.containsOnlyCreatedByDependency());
		
		path = PathImpl.parsePath("/usr/local");
		node = root.createDirectories(path);
		Assert.assertEquals("local", node.getName());
		Assert.assertEquals("/usr/local", node.getPath().getAbsolutePath());
		Assert.assertTrue(node.isDirectory());
		Assert.assertFalse(node.isFile());
		Assert.assertTrue(node.isCreatedByDependency());
		Assert.assertTrue(root.containsOnlyCreatedByDependency());
	}


	/**
	 * <p>
	 * Performs some tests on the {@link FileSystemNodeImpl#getNode(Path)}
	 * constructor.
	 * </p>
	 */
	@Test
	public void testGetNode() {
		FileSystemNode<File> root, node;
		Path path;


		root = new FileSystemNodeImpl<>("");
		Assert.assertNull(root.getParent());
		Assert.assertEquals("", root.getName());
		Assert.assertEquals("/", root.getPath().getAbsolutePath());
		Assert.assertTrue(root.isDirectory());
		Assert.assertFalse(root.isFile());
		Assert.assertTrue(root.isCreatedByDependency());
		Assert.assertTrue(root.containsOnlyCreatedByDependency());
		
		path = PathImpl.parsePath("/usr/local/myprogram");
		root.createDirectories(path);

		path = PathImpl.parsePath("/");
		node = root.getNode(path);
		Assert.assertNotNull(node);
		Assert.assertEquals("", node.getName());
		Assert.assertEquals("/", node.getPath().getAbsolutePath());
		Assert.assertTrue(node.isDirectory());
		Assert.assertFalse(node.isFile());
		Assert.assertTrue(node.isCreatedByDependency());
		Assert.assertTrue(root.containsOnlyCreatedByDependency());

		path = PathImpl.parsePath("/usr");
		node = root.getNode(path);
		Assert.assertNotNull(node);
		Assert.assertEquals("usr", node.getName());
		Assert.assertEquals("/usr", node.getPath().getAbsolutePath());
		Assert.assertTrue(node.isDirectory());
		Assert.assertFalse(node.isFile());
		Assert.assertTrue(node.isCreatedByDependency());
		Assert.assertTrue(root.containsOnlyCreatedByDependency());

		path = PathImpl.parsePath("/usr/local");
		node = root.getNode(path);
		Assert.assertNotNull(node);
		Assert.assertEquals("local", node.getName());
		Assert.assertEquals("/usr/local", node.getPath().getAbsolutePath());
		Assert.assertTrue(node.isDirectory());
		Assert.assertFalse(node.isFile());
		Assert.assertTrue(node.isCreatedByDependency());
		Assert.assertTrue(root.containsOnlyCreatedByDependency());

		path = PathImpl.parsePath("/usr/local/myprogram");
		node = root.getNode(path);
		Assert.assertNotNull(node);
		Assert.assertEquals("myprogram", node.getName());
		Assert.assertEquals("/usr/local/myprogram", node.getPath().getAbsolutePath());
		Assert.assertTrue(node.isDirectory());
		Assert.assertFalse(node.isFile());
		Assert.assertTrue(node.isCreatedByDependency());
		Assert.assertTrue(root.containsOnlyCreatedByDependency());

		path = PathImpl.parsePath("/usr/local/myprogram/lib");
		node = root.getNode(path);
		Assert.assertNull(node);
	}


}
